% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dtgin.R
\name{dtgin}
\alias{dtgin}
\title{Density for the generalized inverse normal distribution truncated to the positive or negative reals}
\usage{
dtgin(
  z,
  alpha,
  mu,
  tau,
  sign = TRUE,
  log = TRUE,
  quasi = FALSE,
  method = "Fortran"
)
}
\arguments{
\item{z}{quantile.}

\item{alpha}{degrees-of-freedom parameter.}

\item{mu}{similar to location parameter, controls asymmetry of the distribution.}

\item{tau}{similar to scale parameter, controls spread of the distribution.}

\item{sign}{logical. \code{TRUE} implies truncation to positive numbers (\code{z} > 0)
and \code{FALSE} to negative numbers (\code{z} < 0).}

\item{log}{logical; should the log of the density be returned? Defaults to TRUE.}

\item{quasi}{logical; should the quasi-density value be returned? Defaults to FALSE.}

\item{method}{string with the method used to compute the parabolic cylinder function
in the normalization constant. \code{method = "Fortran"} uses a compiled Fortran version,
which is the default. \code{method = "R"} uses an R translation of this function.}
}
\value{
Numeric scalar with density.
}
\description{
Density for the generalized inverse normal distribution truncated to the positive or negative reals
}
\details{
Currently, only scalars are supported for the quantile and parameter values.
Density is supported on the positive reals (\code{z} > 0) when \code{sign = TRUE} and to
negative reals (\code{z} < 0) when \code{sign = FALSE}. \code{mu} can take any value
in \eqn{(-\infty, \infty)}. Density is only defined for parameter values
\code{alpha} > 1 or \code{tau} > 0, so it is set to 0 outside of these values.
The quasi-density or kernel is the density without the normalization constant,
use \code{quasi = TRUE} for this behavior.
}
\examples{
# Computing (log) truncated densities
dtgin(z = 1, alpha = 3, mu = 1, tau = 1, sign = TRUE, log = TRUE, quasi = FALSE)
dtgin(z = -1, alpha = 3, mu = -1, tau = 1, sign = FALSE, log = TRUE, quasi = FALSE)

# Generalized inverse normal density with alpha = 5, mu = 0, tau = 1
n_values <- 200
z_vals <- seq(-5, 5, length.out = n_values)

# Truncated to positive reals (z > 0)
fz_p <- sapply(z_vals[z_vals > 0], function(z) dtgin(z, 5, 0, 1, TRUE, FALSE))
fz_p <- c(rep(0, n_values - sum(z_vals > 0)), fz_p)
plot(z_vals, fz_p, type = "l", xlab = 'Values', ylab = 'Density')

# Truncated to positive reals (z < 0)
fz_n <- sapply(z_vals[z_vals < 0], function(z) dtgin(z, 5, 0, 1, FALSE, FALSE))
fz_n <- c(fz_n, rep(0, n_values - sum(z_vals < 0)))
plot(z_vals, fz_n, type = "l", xlab = 'Values', ylab = 'Density')

# Both truncated densities together
plot(z_vals, fz_p, type = "l", xlab = 'Values', ylab = 'Density')
lines(z_vals, fz_n, col = 'blue', lty = 2)
legend('topright', legend = c('z > 0', 'z < 0'),
       col = c('black', 'blue'), lty = 1:2)
}
