% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tsraking.R
\name{build_raking_problem}
\alias{build_raking_problem}
\title{Build the elements of raking problems.}
\usage{
build_raking_problem(
  data_df,
  metadata_df,
  data_df_name = deparse1(substitute(data_df)),
  metadata_df_name = deparse1(substitute(metadata_df)),
  alterability_df = NULL,
  alterSeries = 1,
  alterTotal1 = 0,
  alterTotal2 = 0
)
}
\arguments{
\item{data_df}{(mandatory)

Data frame (object of class "data.frame") that contains the time series data to be reconciled. It must minimally
contain variables corresponding to the component series and cross-sectional control totals specified in the
metadata data frame (argument \code{metadata_df}). If more than one observation (period) is provided, the sum of
the provided component series values will also be preserved as part of implicit temporal constraints.}

\item{metadata_df}{(mandatory)

Data frame (object of class "data.frame") that describes the cross-sectional aggregation constraints
(additivity rules) for the raking problem. Two character variables must be included in the metadata data frame:
\code{series} and \code{total1}. Two variables are optional: \code{total2} (character) and \code{alterAnnual} (numeric). The values
of variable \code{series} represent the variable names of the component series in the input time series data frame
(argument \code{data_df}). Similarly, the values of variables \code{total1} and \code{total2} represent the variable names of
the 1\if{html}{\out{<sup>}}st\if{html}{\out{</sup>}} and 2\if{html}{\out{<sup>}}nd\if{html}{\out{</sup>}} dimension cross-sectional control totals in the input time series data
frame. Variable \code{alterAnnual} contains the alterability coefficient for the temporal constraint associated to
each component series. When specified, the latter will override the default alterability coefficient specified
with argument \code{alterAnnual}.}

\item{data_df_name}{(optional)

String containing the value of argument \code{data_df}.

\strong{Default value} is \code{data_df_name = deparse1(substitute(data_df))}.}

\item{metadata_df_name}{(optional)

String containing the value of argument \code{metadata_df}.

\strong{Default value} is \code{metadata_df_name = deparse1(substitute(metadata_df))}.}

\item{alterability_df}{(optional)

Data frame (object of class "data.frame"), or \code{NULL}, that contains the alterability coefficients variables.
They must correspond to a component series or a cross-sectional control total, that is, a variable with the same
name must exist in the input time series data frame (argument \code{data_df}). The values of these alterability
coefficients will override the default alterability coefficients specified with arguments \code{alterSeries},
\code{alterTotal1} and \code{alterTotal2}. When the input time series data frame contains several observations and the
alterability coefficients data frame contains only one, the alterability coefficients are used (repeated) for
all observations of the input time series data frame. Alternatively, the alterability coefficients data frame
may contain as many observations as the input time series data frame.

\strong{Default value} is \code{alterability_df = NULL} (default alterability coefficients).}

\item{alterSeries}{(optional)

Nonnegative real number specifying the default alterability coefficient for the component series values. It
will apply to component series for which alterability coefficients have not already been specified in the
alterability coefficients data frame (argument \code{alterability_df}).

\strong{Default value} is \code{alterSeries = 1.0} (nonbinding component series values).}

\item{alterTotal1}{(optional)

Nonnegative real number specifying the default alterability coefficient for the 1\if{html}{\out{<sup>}}st\if{html}{\out{</sup>}} dimension
cross-sectional control totals. It will apply to cross-sectional control totals for which alterability
coefficients have not already been specified in the alterability coefficients data frame (argument
\code{alterability_df}).

\strong{Default value} is \code{alterTotal1 = 0.0} (binding 1\if{html}{\out{<sup>}}st\if{html}{\out{</sup>}} dimension cross-sectional control totals)}

\item{alterTotal2}{(optional)

Nonnegative real number specifying the default alterability coefficient for the 2\if{html}{\out{<sup>}}nd\if{html}{\out{</sup>}} dimension
cross-sectional control totals. It will apply to cross-sectional control totals for which alterability
coefficients have not already been specified in the alterability coefficients data frame (argument
\code{alterability_df}).

\strong{Default value} is \code{alterTotal2 = 0.0} (binding 2\if{html}{\out{<sup>}}nd\if{html}{\out{</sup>}} dimension cross-sectional control totals).}
}
\value{
A list with the elements of the raking problem (excluding the implicit temporal totals):
\itemize{
\item \code{x}        : vector of component series initial values
\item \code{c_x}      : vector of component series alterability coefficients
\item \code{comp_cols}: vector of component series (column) names
\item \code{g}        : vector of cross-sectional total initial values
\item \code{c_g}      : vector of cross-sectional total alterability coefficients
\item \code{tot_cols} : vector of cross-sectional total (column) names
\item \code{G}        : cross-sectional total aggregation matrix (\code{g = G \%*\% x})
}
}
\description{
\if{html,text}{(\emph{version française: 
\url{https://StatCan.github.io/gensol-gseries/fr/reference/build_raking_problem.html}})}

This function is used internally by \code{\link[=tsraking]{tsraking()}} to build the elements of the raking problem. It can also
be useful to derive the cross-sectional (marginal) totals of the raking problem manually (outside of the \code{\link[=tsraking]{tsraking()}}
context).
}
\details{
See \code{\link[=tsraking]{tsraking()}} for a detailed description of \emph{time series raking} problems.

The returned raking problem elements do not include the implicit component series temporal totals
when applicable (i.e., elements \code{g} and \code{G} only contain the cross-sectional totals info).

When the input data contains multiple periods (temporal total preservation scenario), raking problem
elements \code{x}, \code{c_x}, \code{g}, \code{c_g} and \code{G} are constructed \emph{column by column} (in "column-major order"),
corresponding to the default behaviour of R for converting objects of class "matrix" into vectors.

Note: argument validation is not performed here; it is (bluntly) assumed that the function is called
by \code{\link[=tsraking]{tsraking()}} where a thorough validation of the arguments is done.
}
\examples{
# Derive the 5 marginal totals of a 2 x 3 two-dimensional data cube using `tsraking()` 
# metadata.

my_metadata <- data.frame(series = c("A1", "A2", "A3",
                                     "B1", "B2", "B3"),
                          total1 = c(rep("totA", 3),
                                     rep("totB", 3)),
                          total2 = rep(c("tot1", "tot2", "tot3"), 2))
my_metadata

# 6 periods of data with marginal totals set to `NA` (they MUST exist in the input data 
# but can be `NA`).
my_data <- data.frame(A1 = c(12, 10, 12,  9, 15,  7),
                      B1 = c(20, 21, 15, 17, 19, 18),
                      A2 = c(14,  9,  8,  9, 11, 10),
                      B2 = c(20, 29, 20, 24, 21, 17),
                      A3 = c(13, 15, 17, 14, 16, 12),
                      B3 = c(24, 20, 30, 23, 21, 19),
                      tot1 = rep(NA, 6),
                      tot2 = rep(NA, 6),
                      tot3 = rep(NA, 6),
                      totA = rep(NA, 6),
                      totB = rep(NA, 6))

# Get the raking problem elements.
p <- build_raking_problem(my_data, my_metadata)
str(p)

# Calculate the 5 marginal totals for all 6 periods.
my_data[p$tot_cols] <- p$G \%*\% p$x
my_data
}
\seealso{
\code{\link[=tsraking]{tsraking()}} \code{\link[=build_balancing_problem]{build_balancing_problem()}}
}
