% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ts_to_bmkDF.R
\name{ts_to_bmkDF}
\alias{ts_to_bmkDF}
\title{Convert a "ts" object to a benchmarks data frame}
\usage{
ts_to_bmkDF(
  in_ts,
  ind_frequency,
  discrete_flag = FALSE,
  alignment = "b",
  bmk_interval_start = 1,
  startYr_cName = "startYear",
  startPer_cName = "startPeriod",
  endYr_cName = "endYear",
  endPer_cName = "endPeriod",
  val_cName = "value"
)
}
\arguments{
\item{in_ts}{(mandatory)

Time series (object of class "ts" or "mts") to be converted.}

\item{ind_frequency}{(mandatory)

Integer specifying the frequency of the indicator (high frequency) series for which the benchmarks (low
frequency series) are related to. The frequency of a time series corresponds to the maximum number of periods
in a year (e.g., 12 for a monthly data, 4 for a quarterly data, 1 for annual data).}

\item{discrete_flag}{(optional)

Logical argument specifying whether the benchmarks correspond to discrete values (anchor points covering a single
period of the indicator series, e.g., end of year stocks) or not. \code{discrete_flag = FALSE} defines non-discrete
benchmarks, i.e., benchmarks that cover several periods of the indicator series (e.g. annual benchmarks cover
4 quarters or 12 months, quarterly benchmarks cover 3 months, etc.).

\strong{Default value} is \code{discrete_flag = FALSE}.}

\item{alignment}{(optional)

Character identifying the alignment of discrete benchmarks (argument \code{discrete_flag = TRUE}) in the benchmark
(low frequency series) interval coverage window:
\itemize{
\item \code{alignment = "b"}: beginning of the benchmark interval window (first period)
\item \code{alignment = "e"}: end of the benchmark interval window (last period)
\item \code{alignment = "m"}: middle of the benchmark interval window (middle period)
}

This argument has no effect for non-discrete benchmarks (\code{discrete_flag = FALSE}).

\strong{Default value} is \code{alignment = "b"}.}

\item{bmk_interval_start}{(optional)

Integer in the [1 .. \code{ind_frequency}] interval specifying the period (cycle) of the indicator (high frequency)
series at which the benchmark (low frequency series) interval window starts. E.g., annual benchmarks corresponding to
fiscal years defined from April to March of the following year would be specified with \code{bmk_interval_start = 4}
for a monthly indicator series (\code{ind_frequency = 12}) and \code{bmk_interval_start = 2} for a quarterly indicator
series (\code{ind_frequency = 4}).

\strong{Default value} is \code{bmk_interval_start = 1}.}

\item{startYr_cName, startPer_cName, endYr_cName, endPer_cName}{(optional)

Strings specifying the name of the numeric variables (columns) in the output data frame that will define the
benchmarks coverage, i.e., the starting and ending year and period (cycle) identifiers.

\strong{Default values} are \code{startYr_cName = "startYear"}, \code{startPer_cName = "startPeriod"}
\code{endYr_cName = "endYear"} and \code{endPer_cName   = "endPeriod"}.}

\item{val_cName}{(optional)

String specifying the name of the numeric variable (column) in the output data frame that will contain the
benchmark values. This argument has no effect for "mts" objects (benchmark variable names are automatically
inherited from the "mts" object).

\strong{Default value} is \code{val_cName = "value"}.}
}
\value{
The function returns a data frame with five or more variables:
\itemize{
\item Benchmark coverage starting year, type numeric (see argument \code{startYr_cName})
\item Benchmark coverage starting period (cycle), type numeric (see argument \code{startPer_cName})
\item Benchmark coverage ending year, type numeric (see argument \code{endtYr_cName})
\item Benchmark coverage ending period (cycle), type numeric (see argument \code{endPer_cName})
\item One ("ts" object) or many ("mts" object) benchmark data variable(s), type numeric (see argument \code{val_cName})
}

Note: the function returns a "data.frame" object than can be explicitly coerced to another type of object
with the appropriate \verb{as*()} function (e.g., \code{tibble::as_tibble()} would coerce it to a tibble).
}
\description{
\if{html,text}{(\emph{version française: 
\url{https://StatCan.github.io/gensol-gseries/fr/reference/ts_to_bmkDF.html}})}

Convert a "ts" (or "mts") object into a benchmarks data frame for the benchmarking functions with five or more variables
(columns):
\itemize{
\item four (4) for the benchmark coverage
\item one (1) for each benchmark time series
}

For discrete benchmarks (anchor points covering a single period of the indicator series, e.g., end of year
stocks), specify \code{discrete_flag = TRUE} and \code{alignment = "b"}, \code{"e"} or \code{"m"}.
}
\examples{
# Annual and quarterly time series
my_ann_ts <- ts(1:5 * 100, start = 2019, frequency = 1)
my_ann_ts
my_qtr_ts <- ts(my_ann_ts, frequency = 4)
my_qtr_ts


# Annual benchmarks for a monthly indicator series
ts_to_bmkDF(my_ann_ts, ind_frequency = 12)

# Annual benchmarks for a quarterly indicator series
ts_to_bmkDF(my_ann_ts, ind_frequency = 4)

# Quarterly benchmarks for a monthly indicator series
ts_to_bmkDF(my_qtr_ts, ind_frequency = 12)

# Start of year stocks for a quarterly indicator series
ts_to_bmkDF(my_ann_ts, ind_frequency = 4, 
            discrete_flag = TRUE)

# End of quarter stocks for a monthly indicator series
ts_to_bmkDF(my_qtr_ts, ind_frequency = 12, 
            discrete_flag = TRUE, alignment = "e")

# April to March annual benchmarks for a ...
# ... monthly indicator series
ts_to_bmkDF(my_ann_ts, ind_frequency = 12, 
            bmk_interval_start = 4)
# ... quarterly indicator series
ts_to_bmkDF(my_ann_ts, ind_frequency = 4, 
            bmk_interval_start = 2)

# End-of-year (April to March) stocks for a ...
# ... monthly indicator series
ts_to_bmkDF(my_ann_ts, ind_frequency = 12, 
            discrete_flag = TRUE, alignment = "e", bmk_interval_start = 4)
# ... quarterly indicator series
ts_to_bmkDF(my_ann_ts, ind_frequency = 4,
            discrete_flag = TRUE, alignment = "e", bmk_interval_start = 2)

# Custom name for the benchmark data variable (column)
ts_to_bmkDF(my_ann_ts, ind_frequency = 12,
            val_cName = "bmk_val")

# Multiple time series: argument `val_cName` ignored
# (the "mts" object column names are always used)
ts_to_bmkDF(ts.union(ser1 = my_ann_ts, ser2 = my_ann_ts / 10), ind_frequency = 12,
            val_cName = "useless_column_name")
}
\seealso{
\code{\link[=ts_to_tsDF]{ts_to_tsDF()}} \code{\link[=stack_bmkDF]{stack_bmkDF()}} \code{\link[=benchmarking]{benchmarking()}} \code{\link[=stock_benchmarking]{stock_benchmarking()}} \link{time_values_conv}
}
