% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hbm.R
\name{hbm}
\alias{hbm}
\title{hbm : Hierarchical Bayesian Small Area Models}
\usage{
hbm(
  formula,
  hb_sampling = "gaussian",
  hb_link = "identity",
  link_phi = "log",
  re = NULL,
  sre = NULL,
  sre_type = NULL,
  car_type = NULL,
  sar_type = NULL,
  M = NULL,
  data,
  prior = NULL,
  handle_missing = NULL,
  m = 5,
  control = list(),
  chains = 4,
  iter = 4000,
  warmup = floor(iter/2),
  cores = 1,
  sample_prior = "no",
  ...
)
}
\arguments{
\item{formula}{Formula specifying the model structure of auxiliary variables and direct estimates
The formula must be provided as a \code{brmsformula} or \code{formula} object. For multivariate models with multiple
auxiliary variables, use the \code{+} operator to combine multiple \code{bf()} formulas.
Example: \code{formula(y ~ x1 + x2 + x3)}, \code{bf(y ~ x1 + x2 + x3)}, or \code{bf(y | mi() ~ mi(x1)) + bf(x1 | mi() ~ x2)}}

\item{hb_sampling}{A character string naming the distribution family of the response variable to be
used in the model (e.g., "gaussian", "bernoulli", "poisson")}

\item{hb_link}{A specification for the model link function. This can be a name/expression or
character string. See the ’Details’ section for more information on link functions supported by each family.}

\item{link_phi}{Link function for the second parameter (phi), typically representing precision, shape, or dispersion
depending on the family used (e.g., "log", "identity")}

\item{re}{Random effects formula specifying the grouping structure in the data.
For example, re = ~(1|area), where "area" is the grouping variable or cluster ID indicating
that observations within the same area share a common random effect. If not specified,
each row will be treated as its own group, meaning a separate random effect is estimated for each observation.}

\item{sre}{An optional grouping factor mapping observations to spatial locations.
If not specified, each observation is treated as a separate location.
It is recommended to always specify a grouping factor to allow for handling of new data in postprocessing methods.}

\item{sre_type}{Determines the type of spatial random effect used in the model. The function currently supports "sar" and "car"}

\item{car_type}{Type of the CAR structure. Currently implemented are "escar" (exact sparse CAR), "esicar" (exact sparse intrinsic CAR),
"icar" (intrinsic CAR), and "bym2".}

\item{sar_type}{Type of the SAR structure. Either "lag" (for SAR of the response values) or
"error" (for SAR of the residuals).}

\item{M}{The M matrix in SAR is a spatial weighting matrix that shows the spatial relationship between locations with certain
weights, while in CAR, the M matrix is an adjacency matrix that only contains 0 and 1 to show the proximity between locations.
SAR is more focused on spatial influences with different intensities, while CAR is more on direct adjacency relationships.
If sre is specified, the row names of M have to match the levels of the grouping factor}

\item{data}{Dataset used for model fitting}

\item{prior}{Priors for the model parameters (default: \code{NULL}).
Should be specified using the \code{brms::prior()} function or a list of such objects.
For example, \code{prior = prior(normal(0, 1), class = "b")} sets a Normal(0,1) prior on the regression coefficients.
Multiple priors can be combined using \code{c()}, e.g.,
\code{prior = c(prior(normal(0, 1), class = "b"), prior(exponential(1), class = "sd"))}.
If \code{NULL}, default priors from \code{brms} will be used.}

\item{handle_missing}{Mechanism to handle missing data (NA values) to ensure model stability and avoid estimation errors.
Three approaches are supported.
The \code{"deleted"} approach performs complete case analysis by removing all rows with any missing values before model fitting.
This is done using a simple filter such as \code{complete.cases(data)}.
It is recommended when the missingness mechanism is Missing Completely At Random (MCAR).
The \code{"multiple"} approach applies multiple imputation before model fitting.
Several imputed datasets are created (e.g., using the \code{mice} package or the \code{brm_multiple()} function in \code{brms}),
the model is fitted separately to each dataset, and the results are combined.
This method is suitable when data are Missing At Random (MAR).
The \code{"model"} approach uses model-based imputation within the Bayesian model itself.
Missing values are incorporated using the \code{mi()} function in the model formula (e.g., \code{y ~ mi(x1) + mi(x2)}),
allowing the missing values to be jointly estimated with the model parameters.
This method also assumes a MAR mechanism and is applicable only for continuous variables.
If data are suspected to be Missing Not At Random (MNAR), none of the above approaches directly apply.
Further exploration, such as explicitly modeling the missingness process or conducting sensitivity analyses, is recommended.}

\item{m}{Number of imputations to perform when using the \code{"multiple"} approach for handling missing data (default: 5).
This parameter is only used if \code{handle_missing = "multiple"}.
It determines how many imputed datasets will be generated.
Each imputed dataset is analyzed separately, and the posterior draws are then combined to account for both within-imputation and between-imputation variability,
following Rubin’s rules. A typical choice is between 5 and 10 imputations, but more may be needed for higher missingness rates.}

\item{control}{A list of control parameters for the sampler (default: \code{list()})}

\item{chains}{Number of Markov chains (default: 4)}

\item{iter}{Total number of iterations per chain (default: 4000)}

\item{warmup}{Number of warm-up iterations per chain (default: floor(iter/2))}

\item{cores}{Number of CPU cores to use (default: 1)}

\item{sample_prior}{Character. Indicates whether draws from priors should be sampled in addition to posterior draws. The options are:
\code{"no"} (default): Do not draw from priors (only posterior draws are obtained). \code{"yes"}: Draw both from the prior and posterior.
\code{"only"}: Draw solely from the prior, ignoring the likelihood. which allows among others to generate draws from the prior predictive distribution.}

\item{...}{Additional arguments}
}
\value{
A \code{hbmfit} object containing :
\item{model}{Summary of \code{brms} object.}
\item{handle_missing}{Handle missing option used in the model.}
\item{data}{Data passed to the \code{hbm} function. }
}
\description{
This function provide flexible modeling approaches to estimate area-level statistics
while incorporating auxiliary information and spatial structures. This function allows users to fit Bayesian models
using the \code{brms} package and supports Gaussian, Bernoulli, Poisson, and other distributions. It also accommodates
spatial random effects (CAR and SAR) and missing data handling (deletion, model-based imputation, and multiple imputation).
}
\details{
Hierarchical Bayesian Small Area Models
}
\examples{
\donttest{

# Load the example dataset
library(hbsaems)
data("data_fhnorm")

# Prepare the dataset
data <- data_fhnorm

# Fit the Basic Model
model <- hbm(
formula = bf(y ~ x1 + x2 + x3), # Formula model
hb_sampling = "gaussian", # Gaussian family for continuous outcomes
hb_link = "identity", # Identity link function (no transformation)
data = data, # Dataset
chains = 4, # Number of MCMC chains
iter = 4000, # Total MCMC iterations
warmup = 2000, # Number of warmup iterations
cores = 2 # Parallel processing
)
summary(model)

# Fit the Basic Model With Defined Random Effect
model_with_defined_re <- hbm(
formula = bf(y ~ x1 + x2 + x3), # Formula model
hb_sampling = "gaussian", # Gaussian family
hb_link = "identity", # Identity link
re = ~(1 | group), # Defined random effect
data = data,
chains = 4,
iter = 4000,
warmup = 2000,
cores = 2
)
summary(model_with_defined_re)

# Fit the Model with Missing Data
# a. Handling missing by deletion
data_miss <- data
data_miss$y[3:5] <- NA 
model_deleted <- hbm(
formula = bf(y ~ x1 + x2 + x3),
hb_sampling = "gaussian",
hb_link = "identity",
re = ~(1 | group),
data = data,
handle_missing = "deleted",
chains = 4,
iter = 4000,
warmup = 2000,
cores = 2
)
summary(model_deleted)

# b. Handling missing using multiple imputation
model_multiple <- hbm(
formula = bf(y ~ x1 + x2 + x3),
hb_sampling = "gaussian",
hb_link = "identity",
re = ~(1 | group),
data = data,
handle_missing = "multiple",
chains = 4,
iter = 4000,
warmup = 2000,
cores = 2
)
summary(model_multiple)

# c. Handling missing during modeling
data_miss$y[3:5] <- NA 
data_miss$x1[6:7] <- NA
model_model <- hbm(
formula = bf(y | mi() ~ mi(x1) + x2 + x3) +
bf(x1 | mi() ~ x2 + x3),
hb_sampling = "gaussian",
hb_link = "identity",
re = ~(1 | group),
data = data,
handle_missing = "model",
chains = 4,
iter = 4000,
warmup = 2000,
cores = 2
)
summary(model_model)

# Fit the Model with Spatial Effect
# a. CAR (Conditional Autoregressive)
data("adjacency_matrix_car")
adjacency_matrix_car

model_spatial_car <- hbm(
formula = bf(y ~ x1 + x2 + x3 ), 
hb_sampling = "gaussian", 
hb_link = "identity", 
data = data, 
sre = "sre",
sre_type = "car",
M = adjacency_matrix_car,
chains = 4, 
iter = 4000, 
warmup = 2000, 
cores = 2 
)
summary(model_spatial_car)

# b. SAR (Simultaneous Autoregressive)
data("spatial_weight_sar")
spatial_weight_sar

model_spatial_sar <- hbm(
formula = bf(y ~ x1 + x2 + x3 ), 
hb_sampling = "gaussian", 
hb_link = "identity", 
data = data, 
sre_type = "sar",
M = spatial_weight_sar,
chains = 4, 
iter = 4000, 
warmup = 2000, 
cores = 2 
)

}
}
\references{
Rao, J. N. K., & Molina, I. (2015). \emph{Small Area Estimation}. John Wiley & Sons.
Bürkner, P. C. (2017). brms: An R package for Bayesian multilevel models using Stan. \emph{Journal of Statistical Software}, 80(1), 1-28.
}
\author{
Achmad Syahrul Choir, Saniyyah Sri Nurhayati, and Sofi Zamzanah
}
