% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MVSK.R
\name{design_MVSK_portfolio_via_sample_moments}
\alias{design_MVSK_portfolio_via_sample_moments}
\title{Design high-order portfolio based on weighted linear combination of first four moments}
\usage{
design_MVSK_portfolio_via_sample_moments(
  lmd = rep(1, 4),
  X_moments,
  w_init = rep(1/length(X_moments$mu), length(X_moments$mu)),
  leverage = 1,
  method = c("Q-MVSK", "MM", "DC"),
  tau_w = 0,
  gamma = 1,
  zeta = 1e-08,
  maxiter = 100,
  ftol = 1e-05,
  wtol = 1e-04,
  stopval = -Inf
)
}
\arguments{
\item{lmd}{Numerical vector of length 4 indicating the weights of first four moments.}

\item{X_moments}{List of moment parameters, see \code{\link{estimate_sample_moments}()}.}

\item{w_init}{Numerical vector indicating the initial value of portfolio weights.}

\item{leverage}{Number (>= 1) indicating the leverage of portfolio.}

\item{method}{String indicating the algorithm method, must be one of: "Q-MVSK", "MM", "DC".}

\item{tau_w}{Number (>= 0) guaranteeing the strong convexity of approximating function.}

\item{gamma}{Number (0 < gamma <= 1) indicating the initial value of gamma.}

\item{zeta}{Number (0 < zeta < 1) indicating the diminishing paramater of gamma.}

\item{maxiter}{Positive integer setting the maximum iteration.}

\item{ftol}{Positive number setting the convergence criterion of function objective.}

\item{wtol}{Positive number setting the convergence criterion of portfolio weights.}

\item{stopval}{Number setting the stop value of objective.}
}
\value{
A list containing the following elements:
\item{\code{w}}{Optimal portfolio vector.}
\item{\code{cpu_time_vs_iterations}}{Time usage over iterations.}
\item{\code{objfun_vs_iterations}}{Objective function over iterations.}
\item{\code{iterations}}{Iterations index.}
\item{\code{convergence}}{Boolean flag to indicate whether or not the optimization converged.}
\item{\code{moments}}{Moments of portfolio return at optimal portfolio weights.}
}
\description{
Design high-order portfolio based on weighted linear combination of first four moments
(i.e., mean, variance, skewness, and kurtosis):
\preformatted{
  minimize     - lmd1*(w'*mu) + lmd2*(w'*Sigma*w) 
               - lmd3*(w'*Phi*w*w) + lmd4*(w'*Psi*w*w*w)
  subject to   ||w||_1 <= leverage, sum(w) == 1.
}
}
\examples{
library(highOrderPortfolios)
data(X50)

# estimate moments
X_moments <- estimate_sample_moments(X50[, 1:10])

# decide moment weights
xi <- 10
lmd <- c(1, xi/2, xi*(xi+1)/6, xi*(xi+1)*(xi+2)/24)

# portfolio optimization
sol <- design_MVSK_portfolio_via_sample_moments(lmd, X_moments)

}
\references{
R. Zhou and D. P. Palomar, "Solving High-Order Portfolios via Successive Convex Approximation Algorithms," 
in \emph{IEEE Transactions on Signal Processing}, vol. 69, pp. 892-904, 2021.
<doi:10.1109/TSP.2021.3051369>.

X. Wang, R. Zhou, J. Ying, and D. P. Palomar, "Efficient and Scalable High-Order Portfolios Design via Parametric Skew-t Distribution," 
Available in arXiv, 2022. <https://arxiv.org/pdf/2206.02412v1.pdf>.
}
\author{
Rui Zhou and Daniel P. Palomar
}
