% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hlt.R
\name{hlt}
\alias{hlt}
\title{Explanatory and Descriptive Higher-Order Item Response Theory 
(Latent Trait Theory)}
\usage{
hlt(
  x,
  z = NULL,
  id,
  iter,
  burn = iter/2,
  delta,
  type = "2p",
  start = list(list(lambda = c(), theta = c(), delta = c(), alpha = c(), beta = c())),
  nchains = 1,
  progress = TRUE,
  verbose = FALSE
)
}
\arguments{
\item{x}{matrix of item responses. Responses must be integers where the 
lowest value is 0 and the highest value is the maximum possible response
for the item with no gaps. If a question is asked with 5 possible responses,
then the possible values should be c(0,1,2,3,4). For binary items, use
c(0,1).}

\item{z}{centered numeric matrix of predictors for the latent regression. 
Default is `z = NULL` so that no regression is performed. All columns
of this matrix must be numeric. For binary items, use the values c(0,1).
For continuous items, center the values on the mean and divide by the 
standard deviation (normalized). For factors with more than two levels, 
recode into multiple columns of c(0,1).}

\item{id}{I.D. vector indexing first-order latent dimension membership
for each of the first-order latent dimensions. We index starting from zero,
not one. If there are three first-order .
latent dimensions with 5 questions per dimension, then the vector will look
like c(0,0,0,0,0,1,1,1,1,1,2,2,2,2,2).}

\item{iter}{number of total iterations.}

\item{burn}{number of burn in iterations.}

\item{delta}{tuning parameter for Metropolis-Hanstings algorithm. Alter 
delta until acceptance.ratio =~ 0.234.}

\item{type}{type of Partial Credit Model to fit. If the partial credit model
is desired (i.e. all alpha parameters = 1), then choose `type = "1p"`.
If the Generalized Parial Credit Model is desired, then choose
`type = "2p"`. The default is `type = "2p"`.}

\item{start}{starting values for the Metropolis-Hastings algorithm.}

\item{nchains}{number of independent MCMC chains. Default is `nchains = 1`.}

\item{progress}{boolean, show progress bar? Defaults to TRUE.}

\item{verbose}{print verbose messages. Defaults to `FALSE`.
Provide a `list` with the following named arguments:
`list(lambda = c(), theta = c(), delta = c(), alpha = c(), beta = c())`
\itemize{
  \item{lambda - }{vector of starting values for the latent factor loadings.}
  \item{theta - }{vector of starting values for the abilities.}
  \item{delta - }{vector of starting values for the difficulties.}
  \item{alpha - }{vector of starting values for the slope parameters.}
  \item{beta - }{vector of starting values for the latent regression parameters}
}
If you choose specify starting values, then the lengths of the starting value
vectors must match the number of parameters in the model.}
}
\value{
A `list` containing:

\itemize{
\item{post}{ - A `matrix` of posterior estimates. Rows are the draws and columns 
are the named parameters.}
\item{accept.rate}{ - acceptance rate of MH algorithm}
\item{theta}{ - `matrix` of mean (first column) and standard deviation 
(second column) of posterior estimates of ability parameters}
\item{nT}{ - number of latent factors estimated}
\item{args}{ - returns the arguments to hlt}
}
}
\description{
Fit a higher-order item response theory model under the generalized 
partial credit measurement model. The goal is to explain multiple latent dimensions
by a single higher-order dimension. We extend this model with an option to perform regression 
on the general latent dimension.
}
\examples{

# set seed
set.seed(153)

# load the asti data set
data("asti")

# shift responses to range from 0 instead of 1
x = as.matrix(asti[, 1:25]) - 1

# subset and transform predictor data
z = asti[, 26:27]
z[, 1] = (z[, 1] == "students") * 1
z[, 2] = (z[, 2] == "male") * 1
z = as.matrix(z)

# specify which items from x belong to each domain
id = c(0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3,3,3,3,4,4,4,4,4,4)

# fit the model
mod = hlt(x, z = z, id = id, iter = 20, burn = 10, delta = 0.002)

mod$accept.rate # ideally 0.234

\donttest{
plot(mod, param = "lambda1", type = "trace")
plot(mod, param = "lambda2", type = "trace")
plot(mod, param = "lambda3", type = "trace")
plot(mod, param = "a1", type = "trace")
plot(mod, param = "d2_3", type = "trace")
plot(mod, param = "beta1", type = "trace")

plot(mod, item = 1, type = "icc")
plot(mod, item = 2, type = "icc")
plot(mod, item = 3, type = "icc")
plot(mod, item = 4, type = "icc")
plot(mod, item = 5, type = "icc")
plot(mod, item = 6, type = "icc")
plot(mod, item = 7, type = "icc", min = -10, max = 10)

summary(mod, param = "all")
summary(mod, param = "delta", digits = 2)
summary(mod, param = "lambda")
summary(mod, param = "alpha")
summary(mod, param = "delta")
summary(mod, param = "theta", dimension = 1)
summary(mod, param = "theta", dimension = 2)
summary(mod, param = "theta", dimension = 3)
summary(mod, param = "theta", dimension = 4)

# start from a previous run's solution
post = tail(mod$post, 1)
start = list(lambda = post[1, c("lambda1", "lambda2", "lambda3", "lambda4", "lambda5")],
             theta = mod$theta_mean, 
             delta = post[1, grepl("^[d]", colnames(post))], 
             alpha = post[1, paste0("a", 1:25)], 
             beta = post[1, c("beta1", "beta2")])
             
mod = hlt(x, z = z, id = id, start = start, iter = 20, burn = 10, delta = 0.002)


}

}
