#' Sainte-Lague formula
#'
#' @rdname ElecFuns
#' @param v Matrix with candidates/parties (the interpretation depends
#' on the function) in the columns. Each cell has the number of votes cast for
#' each candidate/party. For AV and STV, the matrix should have ranked votes, 
#' with each rank in a separate row. 
#' @param m Number of seats to be awarded. 
#' @param threshold Proportion of votes that a party (or candidate) needs to
#' surpass in order to be eligible to receive seats.
#' @param ... Additional optional arguments (currently ignored).
#'
#' @section Details: The saintelague function works with the same procedure as 
#' the dhondt function, but in this case the sequence of numbers used for the
#' division is only comprised by odd numbers (1, 3, 5, 7 and so on). It uses an
#' amount of odd numbers equal to m. 

#' @export
#'
#' @examples
#' ## Sainte-Lague without threshold: 
#' 
#' saintelague(v=example, m=3)
#' 
#' ## Sainte-Lague with 20% threshold: 
#' 
#' saintelague(v=example, m=3, threshold=0.2) 
#' 
saintelague  <- function(v,
                         m,
                         threshold = 0.0,
                         ...){
  if(nrow(v) > 1){
    stop("Sainte-Lague undefined for ranked votes.")
  }
  if(threshold > max(v)/sum(v)){
    stop("Threshold is higher than maximum proportion of votes")
  }
  # apply threshold
  if(sum(prop.table(v[1, ]) < threshold) != ncol(v)){
    v[1, prop.table(v[1, ]) < threshold]  <- 0         
  }    
  divisors = 2 * 1:m - 1
  div_table <- outer(c(v), divisors, "/")
  hi_quot <- order(c(div_table), decreasing = TRUE)[1:m]
  seats <- table(factor(arrayInd(hi_quot, dim(div_table))[,1], levels = 1:ncol(v)))
  names(seats) <- colnames(v)
  return (seats)
}