% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/iCoxph.R
\name{iCoxph}
\alias{iCoxph}
\title{Integrative Cox Model for Uncertain Event Times}
\usage{
iCoxph(
  formula,
  data,
  subset,
  na.action,
  contrasts = NULL,
  start = iCoxph.start(),
  control = iCoxph.control(),
  ...
)
}
\arguments{
\item{formula}{\code{Survi} object specifying the covariates and response
variable in the model, such as \code{Survi(ID, time, event) ~ x1 + x2}.}

\item{data}{An optional data frame, list, or environment that contains the
covariates and response variables included in the model. If not found in
data, the variables are taken from \code{environment(formula)}, usually
the environment from which this function is called.}

\item{subset}{An optional logical vector specifying a subset of observations
to be used in the fitting process.}

\item{na.action}{An optional function that indicates what should the
procedure do if the data contains \code{NA}s.  The default is set by the
na.action setting of \code{\link[base]{options}}.  The "factory-fresh"
default is \code{\link[stats]{na.omit}}.  Other possible values include
\code{\link[stats]{na.fail}}, \code{\link[stats]{na.exclude}}, and
\code{\link[stats]{na.pass}}.  \code{help(na.fail)} for details.}

\item{contrasts}{An optional list, whose entries are values (numeric
matrices or character strings naming functions) to be used as
replacement values for the contrasts replacement function and whose
names are the names of columns of data containing factors.  See
\code{contrasts.arg} of \code{\link[stats]{model.matrix.default}} for
details.}

\item{start}{A list returned by function \code{\link{iCoxph.start}}
specifying starting values of the parameters to be estimated in the
model. Please refer to the arguments of \code{\link{iCoxph.start}} for
the available parameters.}

\item{control}{A list returned by function \code{\link{iCoxph.control}}
specifying control parameters for the model estimation procedure.
Please refer to the arguments of \code{\link{iCoxph.control}} for the
available parameters.}

\item{...}{Other arguments for future usage.  A warning will be thrown if
any invalid argument is specified.}
}
\value{
An \code{\link{iCoxph-class}} object, whose slots include
\itemize{
    \item \code{call}: Function call.
    \item \code{formula}: Formula used in the model fitting.
    \item \code{data}: A processed data frame used for model fitting.
    \item \code{nObs}: Number of observation.
    \item \code{estimates}:
        \itemize{
            \item \code{beta}: Coefficient estimates.
            \item \code{pi}: Estimated parameters in prior multinomial
                distribution indicating the probabilities of corresponding
                record being true.
            \item \code{baseline}: A data frame that contains estimated
                baseline hazard function with columns named \code{time} and
                \code{h0}.
        }
    \item \code{start}: The initial guesses \code{beta_mat} and
        \code{pi_mat} specified for the parameters to be estimated,
        and the set of initial guess \code{beta_start} and \code{pi_start}
        that resulted in the largest objective function, i.e.,
        the observed-data likelihood function.
    \item \code{control}: The control list specified for model fitting.
    \item \code{na.action}: The procedure specified to deal with
        missing values in the covariate.
    \item \code{xlevels}: A list that records the levels in
        each factor variable.
    \item \code{contrasts}: Contrasts specified and used for each
        factor variable.
    \item \code{convergeCode}: \code{code} returned by function
        \code{\link[stats]{nlm}}, which is an integer indicating why the
        optimization process terminated. \code{help(nlm)} for details.
    \item \code{logL}: A numeric vector containing the observed-data
        log-likelihood over iterations.
}
}
\description{
Fit an integrative Cox model proposed by Wang et al. (2020) for
right-censored survival data with uncertain event times due to imperfect
data integration.
}
\examples{
library(intsurv)

## generate simulated survival data with uncertain records
set.seed(123)
simuDat <- simData4iCoxph(nSubject = 200)

## fit the integertive Cox model
fit <- iCoxph(
    Survi(ID, time, event) ~ x1 + x2 + x3 + x4,
    data = simuDat,
    start = iCoxph.start(methods = "nearest"),
    control = iCoxph.control(tol_beta = 1e-5)
)

## estimated covariate coefficients
coef(fit)

## get SE estimates by bootstrap
fit <- bootSe(fit, B = 30)

## summary of the fitted model
summary(fit)
}
\references{
Wang, W., Aseltine, R. H., Chen, K., & Yan, J. (2020). Integrative Survival
Analysis with Uncertain Event Times in Application to A Suicide Risk
Study. \emph{Annals of Applied Statistics}, 14(1), 51--73.
}
\seealso{
\code{\link{iCoxph.start}} and \code{\link{iCoxph.control}}, respectively,
for initial values and control parameters for model fitting;
\code{\link{summary,iCoxph-method}} for summary of fitted model;
\code{\link{coef,iCoxph-method}} for estimated covariate coefficients;
\code{\link{bootSe}} for SE estimates from bootstrap methods.
}
