% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vector.R
\name{vector-locate}
\alias{vector-locate}
\alias{iv_locate_between}
\alias{iv_locate_includes}
\title{Locate relationships between a vector and an iv}
\usage{
iv_locate_between(
  needles,
  haystack,
  ...,
  missing = "equals",
  no_match = NA_integer_,
  remaining = "drop",
  multiple = "all",
  relationship = "none"
)

iv_locate_includes(
  needles,
  haystack,
  ...,
  missing = "equals",
  no_match = NA_integer_,
  remaining = "drop",
  multiple = "all",
  relationship = "none"
)
}
\arguments{
\item{needles, haystack}{\verb{[vector, iv]}

For \verb{iv_*_between()}, \code{needles} should be a vector and \code{haystack} should be
an iv.

For \verb{iv_*_includes()}, \code{needles} should be an iv and \code{haystack} should be
a vector.
\itemize{
\item Each element of \code{needles} represents the value / interval to match.
\item \code{haystack} represents the values / intervals to match against.
}}

\item{...}{These dots are for future extensions and must be empty.}

\item{missing}{\verb{[integer(1) / "equals" / "drop" / "error"]}

Handling of missing values in \code{needles}.
\itemize{
\item \code{"equals"} considers missing values in \code{needles} as exactly equal
to missing values in \code{haystack} when determining if there is a
matching relationship between them.
\item \code{"drop"} drops missing values in \code{needles} from the result.
\item \code{"error"} throws an error if any values in \code{needles} are missing.
\item If a single integer is provided, this represents the value returned in
the \code{haystack} column for values in \code{needles} that are missing.
}}

\item{no_match}{Handling of \code{needles} without a match.
\itemize{
\item \code{"drop"} drops \code{needles} with zero matches from the result.
\item \code{"error"} throws an error if any \code{needles} have zero matches.
\item If a single integer is provided, this represents the value returned in
the \code{haystack} column for values of \code{needles} that have zero matches. The
default represents an unmatched needle with \code{NA}.
}}

\item{remaining}{Handling of \code{haystack} values that \code{needles} never matched.
\itemize{
\item \code{"drop"} drops remaining \code{haystack} values from the result.
Typically, this is the desired behavior if you only care when \code{needles}
has a match.
\item \code{"error"} throws an error if there are any remaining \code{haystack}
values.
\item If a single integer is provided (often \code{NA}), this represents the value
returned in the \code{needles} column for the remaining \code{haystack} values
that \code{needles} never matched. Remaining \code{haystack} values are always
returned at the end of the result.
}}

\item{multiple}{Handling of \code{needles} with multiple matches. For each needle:
\itemize{
\item \code{"all"} returns all matches detected in \code{haystack}.
\item \code{"any"} returns any match detected in \code{haystack} with no guarantees on
which match will be returned. It is often faster than \code{"first"} and
\code{"last"} if you just need to detect if there is at least one match.
\item \code{"first"} returns the first match detected in \code{haystack}.
\item \code{"last"} returns the last match detected in \code{haystack}.
}}

\item{relationship}{Handling of the expected relationship between
\code{needles} and \code{haystack}. If the expectations chosen from the list below
are invalidated, an error is thrown.
\itemize{
\item \code{"none"} doesn't perform any relationship checks.
\item \code{"one-to-one"} expects:
\itemize{
\item Each value in \code{needles} matches at most 1 value in \code{haystack}.
\item Each value in \code{haystack} matches at most 1 value in \code{needles}.
}
\item \code{"one-to-many"} expects:
\itemize{
\item Each value in \code{needles} matches any number of values in \code{haystack}.
\item Each value in \code{haystack} matches at most 1 value in \code{needles}.
}
\item \code{"many-to-one"} expects:
\itemize{
\item Each value in \code{needles} matches at most 1 value in \code{haystack}.
\item Each value in \code{haystack} matches any number of values in \code{needles}.
}
\item \code{"many-to-many"} expects:
\itemize{
\item Each value in \code{needles} matches any number of values in \code{haystack}.
\item Each value in \code{haystack} matches any number of values in \code{needles}.
}

This performs no checks, and is identical to \code{"none"}, but is provided to
allow you to be explicit about this relationship if you know it exists.
\item \code{"warn-many-to-many"} doesn't assume there is any known relationship, but
will warn if \code{needles} and \code{haystack} have a many-to-many relationship
(which is typically unexpected), encouraging you to either take a closer
look at your inputs or make this relationship explicit by specifying
\code{"many-to-many"}.
}

\code{relationship} is applied after \code{filter} and \code{multiple} to allow potential
multiple matches to be filtered out first.

\code{relationship} doesn't handle cases where there are zero matches. For that,
see \code{no_match} and \code{remaining}.}
}
\value{
A data frame containing two integer columns named \code{needles} and \code{haystack}.
}
\description{
This family of functions locates different types of relationships between a
vector and an iv. It works similar to \code{\link[base:match]{base::match()}}, where \code{needles[i]}
checks for a match in all of \code{haystack}. Unlike \code{match()}, \emph{all} matches are
returned, rather than just the first.
\itemize{
\item \code{iv_locate_between()} locates where \code{needles}, a vector, falls between the
bounds of \code{haystack}, an iv.
\item \code{iv_locate_includes()} locates where \code{needles}, an iv, includes the values
of \code{haystack}, a vector.
}

These functions return a two column data frame. The \code{needles} column is an
integer vector pointing to locations in \code{needles}. The \code{haystack} column is
an integer vector pointing to locations in \code{haystack} with a match.
}
\examples{
x <- as.Date(c("2019-01-05", "2019-01-10", "2019-01-07", "2019-01-20"))

y <- iv_pairs(
  as.Date(c("2019-01-01", "2019-01-03")),
  as.Date(c("2019-01-04", "2019-01-08")),
  as.Date(c("2019-01-07", "2019-01-09")),
  as.Date(c("2019-01-10", "2019-01-20")),
  as.Date(c("2019-01-15", "2019-01-20"))
)

x
y

# Find any location where `x` is between the intervals in `y`
loc <- iv_locate_between(x, y)
loc

iv_align(x, y, locations = loc)

# Find any location where `y` includes the values in `x`
loc <- iv_locate_includes(y, x)
loc

iv_align(y, x, locations = loc)

# Drop values in `x` without a match
loc <- iv_locate_between(x, y, no_match = "drop")
loc

iv_align(x, y, locations = loc)

# ---------------------------------------------------------------------------

a <- c(1, NA)
b <- iv(c(NA, NA), c(NA, NA))

# By default, missing values in `needles` are treated as being exactly
# equal to missing values in `haystack`, so the missing value in `a` is
# considered between the missing interval in `b`.
iv_locate_between(a, b)
iv_locate_includes(b, a)

# If you'd like missing values in `needles` to always be considered
# unmatched, set `missing = NA`
iv_locate_between(a, b, missing = NA)
iv_locate_includes(b, a, missing = NA)
}
\seealso{
\link[=relation-locate]{Locating relationships}

\link[=vector-detect]{Detect relationships between a vector and an iv}

\link[=vector-detect-pairwise]{Pairwise detect relationships between a vector and an iv}
}
