% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/leakyIV.R
\name{leakyIV}
\alias{leakyIV}
\title{Bounding Causal Effects with Leaky Instruments}
\usage{
leakyIV(
  dat,
  tau,
  p = 2,
  normalize = TRUE,
  method = "mle",
  approx = TRUE,
  n_boot = NULL,
  bayes = FALSE,
  parallel = TRUE,
  ...
)
}
\arguments{
\item{dat}{Input data. Either (a) an \eqn{n \times d} data frame or matrix of 
observations with columns for treatment, outcome, and candidate instruments; 
or (b) a \eqn{d \times d} covariance matrix over such variables. The latter 
is incompatible with bootstrapping. Note that in either case, the order of
variables is presumed to be treatment (\eqn{X}), outcome (\eqn{Y}), leaky 
instruments (\eqn{Z}).}

\item{tau}{Either (a) a scalar representing the upper bound on the p-norm of 
linear weights on \eqn{Z} in the structural equation for \eqn{Y}; or (b) a 
vector representing upper bounds on the absolute value of each such 
coefficient. See details.}

\item{p}{Power of the norm for the \code{tau} threshold.}

\item{normalize}{Scale candidate instruments to unit variance?}

\item{method}{Estimator for the covariance matrix, if one is not supplied by
\code{dat}. Options include (a) \code{"mle"}, the default; (b) \code{"shrink"}, 
an analytic empirical Bayes solution; or (c) \code{"glasso"}, the graphical 
lasso. See details.}

\item{approx}{Use nearest positive definite approximation if the estimated 
covariance matrix is singular? See details.}

\item{n_boot}{Optional number of bootstrap replicates.}

\item{bayes}{Use Bayesian bootstrap?}

\item{parallel}{Compute bootstrap estimates in parallel? Must register 
backend beforehand, e.g. via \code{doParallel}.}

\item{...}{Extra arguments to be passed to graphical lasso estimator if
\code{method = "glasso"}. Note that the regularization parameter \code{rho}
is required as input, with no default.}
}
\value{
A data frame with columns for \code{ATE_lo} and \code{ATE_hi}, representing
lower and upper bounds of the partial identification interval for the 
causal effect of \eqn{X} on \eqn{Y}. When bootstrapping, the output data 
frame contains \code{n_boot} rows, one for each bootstrap replicate.
}
\description{
Estimates bounds on average treatment effects in linear IV models under 
limited violations of the exclusion criterion.
}
\details{
Instrumental variables are defined by three structural assumptions: they must
be (A1) \emph{relevant}, i.e. associated with the treatment; (A2) 
\emph{unconfounded}, i.e. independent of common causes between treatment and 
outcome; and (A3) \emph{exclusive}, i.e. only affect outcomes through the 
treatment. The \code{leakyIV} algorithm (Watson et al., 2024) relaxes (A3), 
allowing some information leakage from IVs \eqn{Z} to outcomes \eqn{Y} in 
linear systems. While the average treatment effect (ATE) is no longer 
identifiable in this setting, sharp bounds can be computed exactly. 

We assume the following structural equation for the treatment: 
\eqn{X := Z \beta + \epsilon_X}, where the final summand is a noise term that
correlates with the additive noise in the structural equation for the outcome: 
\eqn{Y := Z \gamma + X \theta + \epsilon_Y}. The ATE is given by the 
parameter \eqn{\theta}. Whereas classical IV models require each \eqn{\gamma} 
coefficient to be zero, we permit some direct signal from \eqn{Z} to 
\eqn{Y}. Specifically, \code{leakyIV} provides support for two types of
information leakage: (a) thresholding the \emph{p}-norm of linear weights 
\eqn{\gamma} (scalar \code{tau}); and (b) thresholding the absolute value of 
each \eqn{\gamma} coefficient one by one (vector \code{tau}). 

Numerous methods exist for estimating covariance matrices. \code{leakyIV}
provides support for maximum likelihood estimation (the default), as well as
empirical Bayes shrinkage via \code{corpcor::\link[corpcor]{cov.shrink}} 
(Schäfer & Strimmer, 2005) and the graphical lasso via 
\code{glasso::\link[glasso]{glasso}} (Friedman et al., 2007). These latter 
methods are preferable in high-dimensional settings where sample covariance 
matrices may be unstable or singular. Alternatively, users can pass a 
pre-computed covariance matrix directly as \code{dat}.

Estimated covariance matrices may be singular for some datasets or bootstrap
samples. Behavior in this case is determined by the \code{approx} argument. 
If \code{TRUE}, \code{leakyIV} proceeds with the nearest positive definite 
approximation, computed via Higham's (2002) algorithm (with a warning). If 
\code{FALSE}, bounds are NA (also with a warning).

Uncertainty can be evaluated in leaky IV models using the bootstrap, provided
that covariances are estimated internally and not passed directly. 
Bootstrapping provides a nonparametric sampling distribution for min/max 
values of the ATE. Set \code{bayes = TRUE} to replace the classical bootstrap 
with a Bayesian bootstrap for approximate posterior inference (Rubin, 1981).
}
\examples{
 
set.seed(123)

# Hyperparameters
n <- 200
d_z <- 4
beta <- rep(1, d_z)
gamma <- rep(0.1, d_z)
theta <- 2
rho <- 0.5

# Simulate correlated residuals
S_eps <- matrix(c(1, rho, rho, 1), ncol = 2)
eps <- matrix(rnorm(n * 2), ncol = 2)
eps <- eps \%*\% chol(S_eps)

# Simulate observables from a leaky IV model
z <- matrix(rnorm(n * d_z), ncol = d_z)
x <- z \%*\% beta + eps[, 1]
y <- z \%*\% gamma + x * theta + eps[, 2]
obs <- cbind(x, y, z)

# Run the algorithm
leakyIV(obs, tau = 1)

# With bootstrapping
leakyIV(obs, tau = 1, n_boot = 10)

# With covariance matrix input
S <- cov(obs)
leakyIV(S, tau = 1)


}
\references{
Watson, D., Penn, J., Gunderson, L., Bravo-Hermsdorff, G., Mastouri, A., and
Silva, R. (2024). Bounding causal effects with leaky instruments. \emph{arXiv}
preprint, 2404.04446.

Friedman, J., Hastie, T., and Tibshirani, R. (2007). Sparse inverse 
covariance estimation with the lasso. \emph{Biostatistics}, 9:432-441.

Schäfer, J., and Strimmer, K. (2005). A shrinkage approach to large-scale 
covariance estimation and implications for functional genomics. 
\emph{Statist. Appl. Genet. Mol. Biol.}, 4:32.

Higham, N. (2002). Computing the nearest correlation matrix: A problem from 
finance. \emph{IMA J. Numer. Anal.}, 22:329–343.

Rubin, D.R. (1981). The Bayesian bootstrap. \emph{Ann. Statist.}, 
\emph{9}(1): 130-134.
}
