% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/main.R
\name{leapGP}
\alias{leapGP}
\title{Localized Ensemble of Approximate Gaussian Processes}
\usage{
leapGP(
  X,
  y,
  M0 = ceiling(sqrt(length(y))),
  rho = NA,
  scale = FALSE,
  n = ceiling(sqrt(length(y))),
  start = NA,
  verbose = FALSE,
  justdoit = FALSE,
  ...
)
}
\arguments{
\item{X}{a matrix of training locations (1 row for each training instance)}

\item{y}{a vector of training responses (\code{length(y)} should equal \code{nrow(X)})}

\item{M0}{the number of prediction hubs desired. Defaults to \code{ceiling(sqrt(length(Y)))}.}

\item{rho}{(optional). The parameter controlling time-accuracy tradeoff. Can also be specified during prediction.}

\item{scale}{logical. Do we want the scale parameter to be returned for predictions? If TRUE,
the matrix \eqn{K^{-1}} will be stored for each hub.}

\item{n}{local neighborhood size (for laGP)}

\item{start}{number of starting points for neighborhood (between 6 and n inclusive)}

\item{verbose}{logical. Should status be printed? Deault is FALSE}

\item{justdoit}{logical. Force leapGP to run using specified parameters (may take a long time and/or cause R to crash).}

\item{...}{optional arguments to be passed to \code{laGP()}}
}
\value{
an object of class \code{leapGP} with fields \code{X}, \code{y}, and \code{hubs}.  Also returns scale parameter if \code{scale=TRUE}
}
\description{
Function to train or initialize a leapGP model, as described in Rumsey et al. (2023).
}
\details{
The leapGP is extends the laGP framework of Gramacy & Apley (2015). The methods are equivalent for \code{rho=1},
         but leapGP trades memory for speed when \code{rho < 1}. The method is described in Rumsey et al. (2023) where they demonstrate
         that leapGP is faster than laGP for sequential predictions and is also generally more accurate for some settings of \code{rho}.
}
\examples{
# Generate data
f <- function(x){
   1.3356*(1.5*(1-x[1]) + exp(2*x[1] - 1)*sin(3*pi*(x[1] - 0.6)^2) +
   exp(3*(x[2]-0.5))*sin(4*pi*(x[2] - 0.9)^2))
}
X <- matrix(runif(200), ncol=2)
y <- apply(X, 1, f)

# Generate data for prediction
Xtest <- matrix(runif(200), ncol=2)
ytest <- apply(Xtest, 1, f)

# Train initial model
mod <- leapGP(X, y, M0 = 30)
# Make sequential predictions
pred <- rep(NA, 100)
for(i in 1:100){
  mod <- predict_leapGP(mod, matrix(Xtest[i,], nrow=1), rho=0.9)
  pred[i] <- mod$mean
}
}
\references{
Gramacy, R. B., & Apley, D. W. (2015). Local Gaussian process approximation for large computer experiments. Journal of Computational and Graphical Statistics, 24(2), 561-578.

Rumsey, K. N., Huerta, G., & Derek Tucker, J. (2023). A localized ensemble of approximate Gaussian processes for fast sequential emulation. Stat, 12(1), e576.
}
