% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/matrixNormal_Distribution.R
\name{matrixNormal_Distribution}
\alias{matrixNormal_Distribution}
\alias{dmatnorm}
\alias{pmatnorm}
\alias{rmatnorm}
\title{The Matrix Normal Distribution}
\usage{
dmatnorm(A, M, U, V, tol = .Machine$double.eps^0.5, log = TRUE)

pmatnorm(
  Lower = -Inf,
  Upper = Inf,
  M,
  U,
  V,
  tol = .Machine$double.eps^0.5,
  keepAttr = TRUE,
  algorithm = mvtnorm::GenzBretz(),
  ...
)

rmatnorm(s = 1, M, U, V, tol = .Machine$double.eps^0.5, method = "chol")
}
\arguments{
\item{A}{The numeric n x p matrix that follows the matrix-normal. Value used to calculate the density.}

\item{M}{The mean n x p matrix that is numeric and real. Must contain non-missing values. Parameter of matrix Normal.}

\item{U}{The individual scale n x n real positive-definite matrix (rows). Must contain non-missing values. Parameter of matrix Normal.}

\item{V}{The parameter scale p x p  real positive-definite matrix (columns). Must contain non-missing values. Parameter of matrix Normal.}

\item{tol}{A numeric tolerance level used to check if a matrix is symmetric. That is, a matrix is symmetric if the difference between the matrix and its transpose is between -\code{tol} and \code{tol}.}

\item{log}{Logical; if TRUE, the logarithm of the density is returned.}

\item{Lower}{The n x p matrix of lower limits for CDF.}

\item{Upper}{The n x p matrix of upper limits for CDF.}

\item{keepAttr}{\code{\link{logical}} indicating if
    \code{\link{attributes}} such as \code{error} and \code{msg} should be
    attached to the return value.  The default, \code{TRUE} is back compatible.}

\item{algorithm}{ an object of class \code{\link[mvtnorm]{GenzBretz}},
                    \code{\link[mvtnorm]{Miwa}} or \code{\link[mvtnorm]{TVPACK}}
                    specifying both the algorithm to be used as well as
                    the associated hyper parameters.}

\item{...}{ additional parameters (currently given to \code{GenzBretz} for
              backward compatibility issues). }

\item{s}{The number of observations desired to simulate from the matrix normal. Defaults to 1. Currently has no effect but acts as a placeholder in future releases.}

\item{method}{String specifying the matrix decomposition used to determine the matrix root of the Kronecker product of U and V in \code{rmatnorm}. Possible methods are eigenvalue decomposition ("eigen"), singular value decomposition ("svd"), and Cholesky decomposition ("chol"). The Cholesky (the default) is typically fastest, but not by much though. Passed to **mvtnorm**::\code{\link[mvtnorm]{rmvnorm}}.}
}
\description{
Computes the density (\code{dmatnorm}), calculates the cumulative distribution function (CDF, \code{pmatnorm}), and generates 1 random number (\code{rmatnorm}) from the matrix normal: \deqn{A \sim MatNorm_{n,p}(M, U, V) }.
}
\details{
These functions rely heavily on this following property of matrix normal distribution. Let \code{koch()} refer to the Kronecker product of a matrix. For a n x p matrix \emph{A},
if  \deqn{A \sim MatNorm(M, U, V),} then
\deqn{ vec(A) \sim MVN_{np} (M, Sigma = koch(V,U) ).}

Thus, the probability of \code{Lower} < \code{A} < \code{Upper} in the matrix normal can be found by using the CDF of vec(A), which is given by \code{\link[mvtnorm]{pmvnorm}} function in \pkg{mvtnorm}.  See \code{\link[mvtnorm]{algorithms}} and \code{\link[mvtnorm]{pmvnorm}} for more information.

Also, we can simulate a random matrix \emph{A} from a matrix normal by sampling \emph{vec(A)} from \code{\link[mvtnorm]{rmvnorm}} function in \pkg{mvtnorm}. This matrix \emph{A} takes the rownames from \emph{U} and the colnames from \emph{V}.
}
\note{
Ideally, both scale matrices are positive-definite. If they do not appear to be symmetric, the tolerance should be increased. Since symmetry is checked, the `checkSymmetry` arguments in `mvtnorm::rmvnorm()` are set to FALSE.
}
\section{Calculating Matrix Normal Probabilities}{


 From the \code{mvtnorm} package, three algorithms are available for evaluating normal probabilities: \itemize{
 \item The default is the randomized Quasi-Monte-Carlo procedure by Genz (1992, 1993) and Genz and Bretz (2002) applicable to arbitrary covariance structures and dimensions up to 1000.
  \item For smaller dimensions (up to 20) and non-singular covariance matrices, the algorithm by Miwa et al. (2003) can be used as well.
  \item For two- and three-dimensional problems and semi-infinite integration region, TVPACK implements an interface to the methods described by Genz (2004).
    }
 The \code{...} arguments define the hyper-parameters for GenzBertz algorithm:
 \describe{
\item{maxpts}{maximum number of function values as integer. The internal FORTRAN code always uses a minimum number depending on the dimension.Default 25000.}
\item{abseps}{absolute error tolerance.}
\item{releps}{relative error tolerance as double.}
}
}

\examples{
# Data Used
# if( !requireNamespace("datasets", quietly = TRUE)) { install.packages("datasets")} #part of baseR.
A <- datasets::CO2[1:10, 4:5]
M <- cbind(stats::rnorm(10, 435, 296), stats::rnorm(10, 27, 11))
V <- matrix(c(87, 13, 13, 112), nrow = 2, ncol = 2, byrow = TRUE)
V # Right covariance matrix (2 x 2), say the covariance between parameters.
U <- I(10) # Block of left-covariance matrix ( 84 x 84), say the covariance between subjects.

# PDF
dmatnorm(A, M, U, V)
dmatnorm(A, M, U, V, log = FALSE)

# Generating Probability Lower and Upper Bounds (They're matrices )
Lower <- matrix(rep(-1, 20), ncol = 2)
Upper <- matrix(rep(3, 20), ncol = 2)
Lower
Upper
# The probablity that a randomly chosen matrix A is between Lower and Upper
pmatnorm(Lower, Upper, M, U, V)

# CDF
pmatnorm(Lower = -Inf, Upper, M, U, V)
# entire domain = 1
pmatnorm(Lower = -Inf, Upper = Inf, M, U, V)

# Random generation
set.seed(123)
M <- cbind(rnorm(3, 435, 296), rnorm(3, 27, 11))
U <- diag(1, 3)
V <- matrix(c(10, 5, 5, 3), nrow = 2)
rmatnorm(1, M, U, V)

# M has a different sample size than U; will return an error.
\dontrun{
M <- cbind(rnorm(4, 435, 296), rnorm(4, 27, 11))
rmatnorm(M, U, V)
}

}
\references{
Pocuca, N., Gallaugher, M.P., Clark, K.M., & McNicholas, P.D. (2019). Assessing and Visualizing Matrix Variate Normality. Methodology. <https://arxiv.org/abs/1910.02859>

Gupta, A. K. and D. K. Nagar (1999). Matrix Variate Distributions. Boca Raton: Chapman & Hall/CRC Press.
}
\concept{distribution}
\keyword{distribution}
\keyword{matrixNormal}
