# WARNING - Generated by {fusen} from dev/flat_teaching.Rmd: do not edit by hand

#' Convert Nested Columns Between `data.frame` and `data.table`
#'
#' @description
#' The `convert_nest` function transforms a `data.frame` or `data.table` by converting nested columns 
#' to either `data.frame` or `data.table` format while preserving the original data structure. 
#' Nested columns are automatically detected based on list column identification.
#'
#' @param data A `data.frame` or `data.table` containing nested columns
#' @param to A `character` string specifying the target format. 
#'   Options are `"df"` (data frame) or `"dt"` (data table). Defaults to `"df"`.
#'
#' @details
#' Advanced Nested Column Conversion Features:
#' \itemize{
#'   \item Intelligent automatic detection of all nested (list) columns
#'   \item Comprehensive conversion of entire data structure
#'   \item Non-destructive transformation with data copying
#'   \item Seamless handling of mixed nested structures
#' }
#' 
#' Automatic Detection and Validation:
#' \itemize{
#'   \item Automatically identifies all list columns in the dataset
#'   \item Issues warning if no nested columns are detected
#'   \item Returns original data unchanged when no list columns exist
#'   \item Ensures data integrity through comprehensive checks
#' }
#' 
#' Conversion Strategies:
#' \enumerate{
#'   \item Nested column identification based on `is.list()` detection
#'   \item Preservation of original data integrity through copying
#'   \item Flexible handling of mixed data structures
#'   \item Consistent type conversion across all nested elements
#' }
#'
#' Nested Column Handling:
#' \itemize{
#'   \item Automatically processes all `list` columns
#'   \item Handles `data.table`, `data.frame`, and generic `list` inputs
#'   \item Maintains original column structure and order
#'   \item Prevents in-place modification of source data
#' }
#'
#' @return 
#' A transformed `data.frame` or `data.table` with all nested columns converted to the specified format.
#' If no nested columns are found, returns the original data with a warning.
#'
#' @note
#' Conversion Characteristics:
#' \itemize{
#'   \item Non-destructive transformation of all nested columns
#'   \item Automatic detection eliminates need for manual column specification
#'   \item Supports flexible input and output formats
#'   \item Minimal performance overhead
#' }
#'
#' Warning Conditions:
#' \itemize{
#'   \item Issues warning if no list columns are found in the input data
#'   \item Returns original data unchanged when no conversion is needed
#'   \item Provides clear messages for troubleshooting
#' }
#'
#' @importFrom data.table as.data.table copy
#' @importFrom tibble as_tibble
#' 
#' @export
#' @examples
#' # Example 1: Create nested data structures
#' # Create single nested column
#' df_nest1 <- iris |> 
#'   dplyr::group_nest(Species)     # Group and nest by Species
#'
#' # Create multiple nested columns
#' df_nest2 <- iris |>
#'   dplyr::group_nest(Species) |>  # Group and nest by Species
#'   dplyr::mutate(
#'     data2 = purrr::map(          # Create second nested column
#'       data,
#'       dplyr::mutate, 
#'       c = 2
#'     )
#'   )
#'
#' # Example 2: Convert nested structures
#' # Convert data frame to data table
#' convert_nest(
#'   df_nest1,                      # Input nested data frame
#'   to = "dt"                      # Convert to data.table
#' )
#'
#' # Example 3: Convert data table to data frame
#' dt_nest <- mintyr::w2l_nest(
#'   data = iris,                   # Input dataset
#'   cols2l = 1:2                   # Columns to nest
#' )
#' convert_nest(
#'   dt_nest,                       # Input nested data table
#'   to = "df"                      # Convert to data frame
#' )

convert_nest <- function(data, to = c("df", "dt")) {
  to <- match.arg(to)
  
  # Automatically detect nested columns (list columns)
  nest_cols <- names(data)[sapply(data, is.list)]
  
  # Check if there are any nested columns
  if (length(nest_cols) == 0) {
    warning("No nested (list) columns found in data")
    return(data)
  }
  
  if (to == "df") {
    # If data is data.table, convert to data.frame and copy to avoid modifying original data
    if (inherits(data, "data.table")) {
      data <- as_tibble(copy(data))
    } else if (!inherits(data, "data.frame")) {
      data <- as_tibble(data)
    }
    # Convert each element of nested columns to data.frame
    for (col in nest_cols) {
      data[[col]] <- lapply(data[[col]], function(x) {
        if (inherits(x, "data.table")) {
          as_tibble(copy(x))
        } else if (!inherits(x, "data.frame")) {
          as_tibble(x)
        } else {
          x
        }
      })
    }
  } else if (to == "dt") {
    # If data is not data.table, convert to data.table and copy to avoid modifying original data
    if (!inherits(data, "data.table")) {
      data <- as.data.table(copy(data))
    }
    # Convert each element of nested columns to data.table
    for (col in nest_cols) {
      data[[col]] <- lapply(data[[col]], function(x) {
        if (!inherits(x, "data.table")) {
          as.data.table(copy(x))
        } else {
          x
        }
      })
    }
  }
  
  return(data)
}
