% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sde.sim.R
\name{sde.sim}
\alias{sde.sim}
\title{Simulation of multivariate SDE trajectories.}
\usage{
sde.sim(
  model,
  x0,
  theta,
  dt,
  dt.sim,
  nobs,
  burn = 0,
  nreps = 1,
  max.bad.draws = 5000,
  verbose = TRUE
)
}
\arguments{
\item{model}{An \code{sde.model} object.}

\item{x0}{A vector or a matrix of size \verb{nreps x ndims} of the SDE values at time 0.}

\item{theta}{A vector or matrix of size \verb{nreps x nparams} of SDE parameters.}

\item{dt}{Scalar interobservation time.}

\item{dt.sim}{Scalar interobservation time for simulation.  That is, interally the interobservation time is \code{dt.sim} but only one out of every \code{dt/dt.sim} simulation steps is kept in the output.}

\item{nobs}{The number of SDE observations per trajectory to generate.}

\item{burn}{Scalar burn-in value.  Either an integer giving the number of burn-in steps, or a value between 0 and 1 giving the fraction of burn-in relative to \code{nobs}.}

\item{nreps}{The number of SDE trajectories to generate.}

\item{max.bad.draws}{The maximum number of times that invalid forward steps are proposed.  See Details.}

\item{verbose}{Whether or not to display information on the simulation.}
}
\value{
A list with elements:
\describe{
\item{\code{data}}{An array of size \verb{nobs x ndims x nreps} containing the simulated SDE trajectories.}
\item{\code{params}}{The vector or matrix of parameter values used to generate the data.}
\item{\verb{dt, dt.sim}}{The actual and internal interobservation times.}
\item{\code{nbad}}{The total number of bad draws.}
}
}
\description{
Simulates a discretized Euler-Maruyama approximation to the true SDE trajectory.
}
\details{
The simulation algorithm is a Markov process with \eqn{Y_0 = x_0} and
\deqn{
Y_{t+1} \sim \mathcal{N}(Y_t + \mathrm{dr}(Y_t, \theta) dt_{\mathrm{sim}}, \mathrm{df}(Y_t, \theta) dt_{\mathrm{sim}}),
}{
Y_(t+1) ~ N(Y_t + dr(Y_t, \theta) dt_(sim), df(Y_t, \theta) dt_(sim)),
}
where \eqn{\mathrm{dr}(y, \theta)}{dr(y, \theta)} is the SDE drift function and \eqn{\mathrm{df}(y, \theta)}{df(y, \theta)} is the diffusion function on the \strong{variance} scale.  At each step, a while-loop is used until a valid SDE draw is produced.  The simulation algorithm terminates after \code{nreps} trajectories are drawn or once a total of \code{max.bad.draws} are reached.
}
\examples{
# load pre-compiled model
hmod <- sde.examples("hest")

# initial values
x0 <- c(X = log(1000), Z = 0.1)
theta <- c(alpha = 0.1, gamma = 1, beta = 0.8, sigma = 0.6, rho = -0.8)

# simulate data
dT <- 1/252
nobs <- 2000
burn <- 500
hsim <- sde.sim(model = hmod, x0 = x0, theta = theta,
                dt = dT, dt.sim = dT/10,
                nobs = nobs, burn = burn)

par(mfrow = c(1,2))
plot(hsim$data[,"X"], type = "l", xlab = "Time", ylab = "",
     main = expression(X[t]))
plot(hsim$data[,"Z"], type = "l", xlab = "Time", ylab = "",
     main = expression(Z[t]))
}
