% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/functions_auxiliary.R
\name{ph_test_statistic}
\alias{ph_test_statistic}
\title{Pointwise Hotelling's \eqn{T^2}-test statistic}
\usage{
ph_test_statistic(x, gr_label, h)
}
\arguments{
\item{x}{matrix of observations \eqn{n\times j} (\eqn{n = n_1 + ... + n_k}, \eqn{j} is a number of design time points).}

\item{gr_label}{a vector with group labels; the integer labels (from 1 to a number of groups) should be used.}

\item{h}{contrast matrix. For Dunnett’s and Tukey’s contrasts, it can be created by
the \code{contr_mat()} function from the package \code{GFDmcv} (see examples).}
}
\value{
A vector of values of the pointwise Hotelling's \eqn{T^2}-test statistic.
}
\description{
The function \code{ph_test_statistic()} calculates the pointwise Hotelling's \eqn{T^2}-test statistic.
}
\details{
For details, see the documentation of the \code{multiFANOVA()} function or
the paper Munko et al. (2023).
}
\examples{
# Some of the examples may run some time.

# Canadian weather data set
# There are three samples of mean temperatures for
# fifteen weather stations in Eastern Canada,
# another fifteen in Western Canada, and
# the remaining five in Northern Canada.
library(fda)
data_set <- t(CanadianWeather$dailyAv[,, "Temperature.C"])
k <- 3
gr_label <- rep(c(1, 2, 3), c(15, 15, 5))
# trajectories of mean temperatures
matplot(t(data_set), type = "l", col = gr_label, lty = 1,
        xlab = "Day", ylab = "Temperature (C)",
        main = "Canadian weather data set")
legend("bottom", legend = c("Eastern Canada", "Western Canada", "Northern Canada"),
       col = 1:3, lty = 1)
\donttest{
# Tukey's contrast matrix
h_tukey <- GFDmcv::contr_mat(k, type = "Tukey")
# testing without parallel computing
res <- multiFANOVA(data_set, gr_label, h_tukey)
summary(res, digits = 3)
# plots for pointwise Hotelling's T^2-test statistics
oldpar <- par(mfrow = c(2, 2), mar = c(2, 2, 2, 0.1))
plot(ph_test_statistic(data_set, gr_label, h_tukey), type = "l",
     ylim = c(0, max(ph_test_statistic(data_set, gr_label, h_tukey))),
     main = "Global hypothesis")
plot(ph_test_statistic(data_set, gr_label, matrix(h_tukey[1, ], 1)), type = "l",
     ylim = c(0, max(ph_test_statistic(data_set, gr_label, h_tukey))),
     main = "Contrast 1")
plot(ph_test_statistic(data_set, gr_label, matrix(h_tukey[2, ], 1)), type = "l",
     ylim = c(0, max(ph_test_statistic(data_set, gr_label, h_tukey))),
     main = "Contrast 2")
plot(ph_test_statistic(data_set, gr_label, matrix(h_tukey[3, ], 1)), type = "l",
     ylim = c(0, max(ph_test_statistic(data_set, gr_label, h_tukey))),
     main = "Contrast 3")
par(oldpar)}
\dontshow{
h_tukey <- GFDmcv::contr_mat(k, type = "Tukey")
res <- multiFANOVA(data_set[, 1:10], gr_label, h_tukey, n_boot = 3)
summary(res, digits = 3)
# plots for pointwise Hotelling's T^2-test statistics
oldpar <- par(mfrow = c(2, 2), mar = c(2, 2, 2, 0.1))
plot(ph_test_statistic(data_set, gr_label, h_tukey), type = "l",
     ylim = c(0, max(ph_test_statistic(data_set, gr_label, h_tukey))),
     main = "Global hypothesis")
plot(ph_test_statistic(data_set, gr_label, matrix(h_tukey[1, ], 1)), type = "l",
     ylim = c(0, max(ph_test_statistic(data_set, gr_label, h_tukey))),
     main = "Contrast 1")
plot(ph_test_statistic(data_set, gr_label, matrix(h_tukey[2, ], 1)), type = "l",
     ylim = c(0, max(ph_test_statistic(data_set, gr_label, h_tukey))),
     main = "Contrast 2")
plot(ph_test_statistic(data_set, gr_label, matrix(h_tukey[3, ], 1)), type = "l",
     ylim = c(0, max(ph_test_statistic(data_set, gr_label, h_tukey))),
     main = "Contrast 3")
par(oldpar)}

}
\references{
Dunnett C. (1955) A multiple comparison procedure for comparing several treatments
with a control. Journal of the American Statistical Association 50, 1096-1121.

Munko M., Ditzhaus M., Pauly M., Smaga L., Zhang J.T. (2023) General multiple tests for functional data. Preprint https://arxiv.org/abs/2306.15259

Tukey J.W. (1953) The problem of multiple comparisons. Princeton University.
}
