# Weighted tau-d.
#'
#' Computes weighted tau from Section 2.1.
#' Agresti, A. (1983). Testing marginal homogeneity for ordinal categorical variables.
#' Biometrics, 39(2), 505-510.
#' @param n matrix of observed counts
#' @returns a list containing
#'    tau: value of tau-d coefficient
#'    sigma_tau: SE(tau)
#'    z_tau: z-score for tau
#' @export
Agresti_weighted_tau <- function(n) {
  p <- n / sum(n)
  p_i <- rowSums(p)
  p_j <- colSums(p)
  gamma_i <- cumsum(p_i)
  gamma_j <- cumsum(p_j)

  phi <- matrix(nrow = nrow(n), ncol = ncol(n))
  tau <- sum(p_j * gamma_i) - sum(p_i*gamma_j)
  for (i in 1:nrow(p)) {
    gi <- gamma_i[i]
    if (i > 1) {
      gi <- gi + gamma_i[i - 1]
    }
    for (j in 1:ncol(p)) {
      gj = gamma_j[j]
      if (j > 1) {
        gj <- gj + gamma_j[j - 1]
      }
      phi[i, j] <- gj - gi
    }
  }
  var <- (sum(phi * phi * p) - (sum(phi * p))^2) / sum(n)
  sigma <- sqrt(var)
  list(tau=tau, sigma_tau=sigma, z_tau=tau / sigma)
}


#' Computes the weighted statistics listed in section 2.3.
#'
#' Computes weighted contrast of the two margins.
#' Agresti, A. (1983). Testing marginal homogeneity for ordinal categorical variables.
#' Biometrics, 39(2), 505-510.
#' @param w a vector of weights to be treated as scores
#' @param n matrix of observed counts
#' @returns a list containing
#'    diff: the weighted contrast computed using weights w
#'    sigma_diff: SE(diff)
#'    z_diff: z-score for diff
#' @export
#' @examples
#' weights = c(-3.0, -1.0, 1.0, 3.0)
#' Agresti_w_diff(weights, vision_data)
Agresti_w_diff <- function(w, n) {
  if (length(w)^2 != length(n)) {
    stop(paste("length of data", length(n), "must match squared length of weights", length(n),
               "in agresti::w_diff"))
  }
  n_total <- sum(n)
  p <- n / n_total
  p_i = rowSums(p)
  p_j = colSums(p)
  nr = nrow(n)
  nc = ncol(n)
  if (nc != nr) stop("n has to be square")
  variance <- 0.0
  for (i in 1:nr) {
    for (j in 1:nc) {
      variance  <- variance + p[i, j] * (w[i] - w[j])^2
    }
  }

  m1 <- sum(w * p_i)
  m2 <- sum(w * p_j)
  variance <- (variance - (m1 - m2)^2) / n_total
  list(diff=m1 - m2, sigma_diff=sqrt(variance), z_diff=(m1 - m2) / sqrt(variance))
}
