#' Create manifest of local files/directories to upload
#'
#' @param path a scalar vector of file paths for local files and directories
#' @param recurse determines whether directories in `path` are recursed
#' @return data.frame of file metadata generated by `fs::file_info`, with the
#'   the following additional columns: `remote_path` and `remote_dir`
#' @noRd
.upload_manifest <- function(path, recurse) {
  stopifnot(rlang::is_scalar_vector(path))

  if (fs::is_dir(path)) {
    targets <- fs::dir_ls(path, type = c("file", "directory"), recurse = recurse)
  }

  if (fs::is_file(path) || rlang::is_empty(targets)) {
    targets <- path
  }

  inventory <- fs::file_info(targets)

  # the leaf directory of `path` will be the base destination for uploads
  inventory$remote_path <- fs::path_rel(inventory$path, dirname(path))
  inventory$remote_dir <- ifelse(
    inventory$type == "file",
    dirname(inventory$remote_path),
    identity(inventory$remote_path)
  )

  # drop unused file type levels
  droplevels(inventory)
}

#' Add column containing each file's remote destination
#' @param x OSF upload destination
#' @param manifest Upload manifest with `remote_dir` column
#' @return Upload manifest modified to include a new column (`remote_dest`) that
#'   contains a list of `osf_tbl`s representing the upload destination for each
#'   file.
#' @noRd
.ulm_add_remote_dests <- function(manifest, dest, verbose) {

  # retrieve (or create) each unique remote directory
  remote_dirs <- unique(manifest$remote_dir)
  if (any(remote_dirs != ".")) {
    if (verbose) message("Retrieving upload destinations from OSF")
  }

  remote_dests <- Map(recurse_path,
    path = remote_dirs,
    x = list(dest),
    missing_action = "create",
    verbose = verbose
  )

  # pair local files with the osf_tbls remote destinations
  manifest$remote_dest <- remote_dests[manifest$remote_dir]
  manifest
}

#' Add column containing conflicting files at remote destinations
#' @param x OSF upload destination
#' @param manifest Upload manifest with `remote_dir` column
#' @return Upload manifest modified to include a new column (`remote_dest`) that
#'   contains a list of `osf_tbl`s representing the upload destination for each
#'   file.
#' @noRd
.ulm_add_conflicting_files <- function(manifest, verbose) {

  if (verbose) message("Searching for conflicting files on OSF")

  # list existing files within each remote destination
  remote_dests <- manifest$remote_dest[!duplicated(manifest$remote_dir)]
  remote_files <- map_rbind(
    osf_ls_files,
    x = remote_dests,
    type = "file",
    n_max = Inf,
    verbose = verbose
  )

  # pair local files with their existing remote counterparts
  remote_files <- split(remote_files, get_remote_path(remote_files))
  manifest$remote_file <- remote_files[manifest$remote_path]
  manifest
}
