% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate_C.R
\name{estimate_C}
\alias{estimate_C}
\title{Estimate constraint constant C for continuous data}
\usage{
estimate_C(X, qmax = 8, safety = 1.2)
}
\arguments{
\item{X}{n x p continuous data matrix}

\item{qmax}{Rank for truncated SVD (default 8)}

\item{safety}{Safety parameter for conservative estimation (default 1.2)}
}
\value{
A list containing:
  \item{qmax}{Truncation rank used}
  \item{safety}{Safety parameter applied}
  \item{C_norm_hat}{Original maximum row norm}
  \item{C_est}{Final conservative estimate of C}
  \item{a_norms}{Row norms of factor matrix A}
  \item{b_norms}{Row norms of factor matrix B}
}
\description{
Data-driven estimation of the constraint constant C in alternating maximization algorithm for continuous data
             using truncated SVD approach. This function decomposes the data matrix
             and estimates C based on the maximum row norms.
}
\details{
The function performs the following steps:
1. Computes truncated SVD of X with rank qmax
2. Constructs factor matrices A = U * sqrt(D) and B = V * sqrt(D)
3. Calculates row 2-norms for matrices A and B
4. Takes the maximum norm and multiplies by safety parameter

For count data, it is recommended to transform the data using log(X + 1) before
applying this function.
}
\examples{
# Example 1: Continuous data
set.seed(123)
n <- 100; p <- 50; q <- 3
theta_true <- matrix(runif(n * q), n, q)
A_true <- matrix(runif(p * q), p, q)
X <- theta_true \%*\% t(A_true) + matrix(rnorm(n * p, sd = 0.5), n, p)

# Estimate C
C_result <- estimate_C(X, qmax = 5)
print(C_result$C_est)

# Example 2: Count data (apply log transformation)
lambda <- exp(theta_true \%*\% t(A_true))
X_count <- matrix(rpois(n * p, lambda = as.vector(lambda)), n, p)
X_transformed <- log(X_count + 1)
C_count <- estimate_C(X_transformed, qmax = 5)
print(C_count$C_est)
}
