% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CentSimUG2D.R
\name{edgesCStri}
\alias{edgesCStri}
\title{The edges of the underlying or reflexivity graphs of
the Central Similarity Proximity Catch Digraph
(CS-PCD) for 2D data - one triangle case}
\usage{
edgesCStri(Xp, tri, t, M = c(1, 1, 1), ugraph = c("underlying", "reflexivity"))
}
\arguments{
\item{Xp}{A set of 2D points
which constitute the vertices of the underlying
or reflexivity graphs of the CS-PCD.}

\item{tri}{A \eqn{3 \times 2} matrix with each row
representing a vertex of the triangle.}

\item{t}{A positive real number
which serves as the expansion parameter in CS proximity region.}

\item{M}{A 2D point in Cartesian coordinates
or a 3D point in barycentric coordinates
which serves as a center in the interior of the triangle \code{tri};
default is \eqn{M=(1,1,1)}, i.e.,
the center of mass of \code{tri}.}

\item{ugraph}{The type of the graph based on CS-PCDs,
\code{"underlying"} is for the underlying graph,
and \code{"reflexivity"} is for
the reflexivity graph (default is \code{"underlying"}).}
}
\value{
A \code{list} with the elements
\item{type}{A description of the underlying
or reflexivity graph of the digraph}
\item{parameters}{Parameters of the underlying or reflexivity graph of the digraph,
the center \code{M} used to
construct the edge regions and the expansion parameter \code{t}.}
\item{tess.points}{Tessellation points,
i.e., points on which the tessellation of the study region
is performed, here, tessellation is the support triangle.}
\item{tess.name}{Name of the tessellation points \code{tess.points}}
\item{vertices}{Vertices of the underlying
or reflexivity graph of the digraph, \code{Xp} points}
\item{vert.name}{Name of the data set
which constitutes the vertices of
the underlying or reflexivity graph of the digraph}
\item{LE, RE}{Left and right end points of the edges of
the underlying or reflexivity graph of CS-PCD for 2D data set \code{Xp}
as vertices of the underlying or reflexivity graph of the digraph}
\item{mtitle}{Text for \code{"main"} title
in the plot of the underlying or reflexivity graph of the digraph}
\item{quant}{Various quantities for the underlying
or reflexivity graph of the digraph:
number of vertices, number of partition points,
number of intervals, number of edges, and edge density.}
}
\description{
An object of class \code{"UndPCDs"}.
Returns edges of the underlying or reflexivity graph of CS-PCD
as left and right end points
and related parameters and the quantities of these graphs.
The vertices of these graphs are the data points in \code{Xp}
in the multiple triangle case.

CS proximity regions are constructed
with respect to the triangle \code{tri} with expansion
parameter \eqn{t > 0}, i.e.,
edges may exist only for points inside \code{tri}.
It also provides various descriptions
and quantities about the edges of
the underlying or reflexivity graphs of the CS-PCD
such as number of edges, edge density, etc.

Edge regions are based on center \eqn{M=(m_1,m_2)}
in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)}
in barycentric coordinates in the interior of
the triangle \code{tri};
default is \eqn{M=(1,1,1)}, i.e., the center of mass of \code{tri}.
With any interior center \code{M},
the edge regions are constructed using the extensions
of the lines combining vertices with \code{M}.

See also
(\insertCite{ceyhan:Phd-thesis,ceyhan:stamet2016;textual}{pcds.ugraph}).
}
\examples{
#\donttest{
A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
Tr<-rbind(A,B,C);
n<-10

set.seed(1)
Xp<-pcds::runif.tri(n,Tr)$g

M<-as.numeric(pcds::runif.tri(1,Tr)$g)

t<-1.5

#for underlying graph
Edges<-edgesCStri(Xp,Tr,t,M)
Edges
summary(Edges)
plot(Edges)

#for reflexivity graph
Edges<-edgesCStri(Xp,Tr,t,M,ugraph="r")
Edges
summary(Edges)
plot(Edges)

#can add edge regions
cent<-M
cent.name<-"M"
Ds<-pcds::prj.cent2edges(Tr,M)
L<-rbind(cent,cent,cent); R<-Ds
segments(L[,1], L[,2], R[,1], R[,2], lty=2)

#now we can add the vertex names and annotation
txt<-rbind(Tr,cent,Ds)
xc<-txt[,1]+c(-.02,.02,.02,.02,.03,-.03,-.01)
yc<-txt[,2]+c(.02,.02,.03,.06,.04,.05,-.07)
txt.str<-c("A","B","C","M","D1","D2","D3")
text(xc,yc,txt.str)
#}

}
\references{
\insertAllCited{}
}
\seealso{
\code{\link{edgesCS}}, \code{\link{edgesAStri}}, \code{\link{edgesPEtri}},
and \code{\link[pcds]{arcsCStri}}
}
\author{
Elvan Ceyhan
}
