% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gradient.R
\name{Jacobian}
\alias{Jacobian}
\title{Jacobian matrix computation with parallel capabilities
s
Computes the numerical Jacobian for vector-valued functions. Its columns are
partial derivatives of the function with respect to the input elements.
This function supports both two-sided (central, symmetric) and
one-sided (forward or backward) derivatives. It can utilise parallel processing
to accelerate computation of gradients for slow functions or
to attain higher accuracy faster. Currently, only Mac and Linux are supported
\code{parallel::mclapply()}. Windows support with \code{parallel::parLapply()}
is under development.}
\usage{
Jacobian(
  FUN,
  x,
  elementwise = NA,
  vectorised = NA,
  multivalued = NA,
  deriv.order = 1L,
  side = 0,
  acc.order = 2,
  stencil = NULL,
  h = NULL,
  zero.tol = sqrt(.Machine$double.eps),
  h0 = NULL,
  control = list(),
  f0 = NULL,
  cores = 1,
  preschedule = TRUE,
  cl = NULL,
  func = NULL,
  method = NULL,
  method.args = list(),
  ...
)
}
\arguments{
\item{FUN}{A function returning a numeric scalar or a vector whose derivatives are to be
computed. If the function returns a vector, the output will be a Jacobian.}

\item{x}{Numeric vector or scalar: the point(s) at which the derivative is estimated.
\code{FUN(x)} must be finite.}

\item{elementwise}{Logical: is the domain effectively 1D, i.e. is this a mapping
\eqn{\mathbb{R} \mapsto \mathbb{R}}{R -> R} or
\eqn{\mathbb{R}^n \mapsto \mathbb{R}^n}{R^n -> R^n}. If \code{NA},
compares the output length ot the input length.}

\item{vectorised}{Logical: if \code{TRUE}, the function
is assumed to be vectorised: it will accept a vector of parameters and return
a vector of values of the same length. Use \code{FALSE} or \code{"no"}  for
functions that take vector arguments and return outputs of arbitrary length (for
\eqn{\mathbb{R}^n \mapsto \mathbb{R}^k}{R^n -> R^k} functions). If \code{NA},
checks the output length and assumes vectorisation if it matches the input length;
this check is necessary and potentially slow.}

\item{multivalued}{Logical: if \code{TRUE}, the function is assumed to return vectors longer
than 1. Use \code{FALSE} for element-wise functions. If \code{NA}, attempts inferring it from
the function output.}

\item{deriv.order}{Integer or vector of integers indicating the desired derivative order,
\eqn{\mathrm{d}^m / \mathrm{d}x^m}{d^m/dx^m}, for each element of \code{x}.}

\item{side}{Integer scalar or vector indicating the type of finite difference:
\code{0} for central, \code{1} for forward, and \code{-1} for backward differences.
Central differences are recommended unless computational cost is prohibitive.}

\item{acc.order}{Integer or vector of integers specifying the desired accuracy order
for each element of \code{x}.
The final error will be of the order \eqn{O(h^{\mathrm{acc.order}})}{O(h^acc.order)}.}

\item{stencil}{Optional custom vector of points for function evaluation.
Must include at least \code{m+1} points for the \code{m}-th order derivative.}

\item{h}{Numeric or character specifying the step size(s) for the numerical
difference or a method of automatic step determination (\code{"CR"}, \code{"CRm"},
\code{"DV"}, or \code{"SW"} to be used in \code{\link[=gradstep]{gradstep()}}). The default value is
described in \code{?GenD}.}

\item{zero.tol}{Small positive integer: if \code{abs(x) >= zero.tol}, then, the automatically
guessed step size is relative (\code{x} multiplied by the step), unless an auto-selection
procedure is requested; otherwise, it is absolute.}

\item{h0}{Numeric scalar of vector: initial step size for automatic search with
\code{gradstep()}.}

\item{control}{A named list of tuning parameters passed to \code{gradstep()}.}

\item{f0}{Optional numeric: if provided, used to determine the vectorisation type
to save time. If FUN(x) must be evaluated (e.g. second derivatives), saves one evaluation.}

\item{cores}{Integer specifying the number of CPU cores used for parallel computation.
Recommended to be set to the number of physical cores on the machine minus one.}

\item{preschedule}{Logical: if \code{TRUE}, disables pre-scheduling for \code{mclapply()}
or enables load balancing with \code{parLapplyLB()}. Recommended for functions that
take less than 0.1 s per evaluation.}

\item{cl}{An optional user-supplied \code{cluster} object  (created by \code{makeCluster}
or similar functions). If not \code{NULL}, the code uses \code{parLapply()} (if \code{preschedule}
is \code{TRUE}) or \code{parLapplyLB()} on that cluster on Windows, and \code{mclapply}
(fork cluster) on everything else.}

\item{func}{For compatibility with \code{numDeriv::grad()} only. If instead of
\code{FUN}, \code{func} is used, it will be reassigned to \code{FUN} with a warning.}

\item{method}{For compatibility with \code{numDeriv::grad()} only. Supported values:
\code{"simple"} and \code{"Richardson"}. Non-null values result in a warning.}

\item{method.args}{For compatibility with \code{numDeriv::grad()} only. Check
\code{?numDeriv::grad} for a list of values. Non-empty lists result in a warning.}

\item{...}{Ignored.}
}
\value{
Matrix where each row corresponds to a function output and each column
to an input coordinate. For scalar-valued functions, a warning is issued and
the output is returned as a row matrix.
}
\description{
Jacobian matrix computation with parallel capabilities
s
Computes the numerical Jacobian for vector-valued functions. Its columns are
partial derivatives of the function with respect to the input elements.
This function supports both two-sided (central, symmetric) and
one-sided (forward or backward) derivatives. It can utilise parallel processing
to accelerate computation of gradients for slow functions or
to attain higher accuracy faster. Currently, only Mac and Linux are supported
\code{parallel::mclapply()}. Windows support with \code{parallel::parLapply()}
is under development.
}
\examples{
slowFun <- function(x) {Sys.sleep(0.01); sum(sin(x))}
slowFunVec <- function(x) {Sys.sleep(0.01);
                           c(sin = sum(sin(x)), exp = sum(exp(x)))}
true.g <- cos(1:4)  # Analytical gradient
true.j <- rbind(cos(1:4), exp(1:4)) # Analytical Jacobian
x0 <- c(each = 1, par = 2, is = 3, named = 4)

# Compare computation times
system.time(g.slow <- numDeriv::grad(slowFun, x = x0) - true.g)
system.time(j.slow <- numDeriv::jacobian(slowFunVec, x = x0) - true.j)
system.time(g.fast <- Grad(slowFun, x = x0, cores = 2) - true.g)
system.time(j.fast <- Jacobian(slowFunVec, x = x0, cores = 2) - true.j)
system.time(j.fast4 <- Jacobian(slowFunVec, x = x0, acc.order = 4, cores = 2) - true.j)

# Compare accuracy
rownames(j.slow) <- paste0("numDeriv.jac.", c("sin", "exp"))
rownames(j.fast) <- paste0("pnd.jac.order2.", rownames(j.fast))
rownames(j.fast4) <- paste0("pnd.jac.order4.", rownames(j.fast4))
# Discrepancy
print(rbind(numDeriv.grad = g.slow, pnd.Grad = g.fast, j.slow, j.fast, j.fast4), 2)
# The order-4 derivative is more accurate for functions
# with non-zero third and higher derivatives -- look at pnd.jac.order.4


}
\seealso{
\code{\link[=GenD]{GenD()}}, \code{\link[=Grad]{Grad()}}
}
