#' @title
#' Computing the incomparability relation of a poset.
#'
#' @description
#' Computes the incomparability relation of the input poset.
#'
#' @param poset An object of S4 class POSet.
#' Argument `poset` must be created by using any function contained in the package aimed at building object of S4 class `POSet`
#' (e.g. [POSet()], [LinearPOSet()], [ProductPOSet()], ...) .
#'
#' @return
#' A two-column matrix \eqn{M} (element \eqn{M[i,2]} is incomparable with element \eqn{M[i,1]}).
#'
#' @examples
#' elems <- c("a", "b", "c", "d")
#'
#' dom <- matrix(c(
#'   "a", "b",
#'   "c", "b"
#' ), ncol = 2, byrow = TRUE)
#'
#' pos <- POSet(elements = elems, dom = dom)
#'
#' M <- IncomparabilityRelation(pos)
#'
#' @name IncomparabilityRelation
#' @export IncomparabilityRelation
IncomparabilityRelation <- function(poset) {
  if (!methods::is(poset, "POSet")) {
    stop("poset must be of class POSet")
  }
  tryCatch({
    result <- .Call("_Incomparabilities", poset@ptr)
    return (result)
  }, error = function(err) {
    # error handler picks up where error was generated
    print(paste("MY_ERROR:  ",err))
  }) # END tryCatch
}
