% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/oneill_1972.R
\name{oneill_1972}
\alias{oneill_1972}
\title{O'Neill model for fitting thermal performance curves}
\usage{
oneill_1972(temp, rmax, ctmax, topt, q10)
}
\arguments{
\item{temp}{temperature in degrees centigrade}

\item{rmax}{maximum rate at optimum temperature}

\item{ctmax}{high temperature (ºC) at which rates become negative}

\item{topt}{optimum temperature (ºC)}

\item{q10}{defines the fold change in performance as a result of increasing the temperature by 10 ºC}
}
\value{
a numeric vector of rate values based on the temperatures and parameter values provided to the function
}
\description{
O'Neill model for fitting thermal performance curves
}
\details{
Equation:
\deqn{rate = r_{max} \cdot \bigg(\frac{ct_{max} - temp}{ct_{max} - t_{opt}}\bigg)^{x} \cdot exp^{x \cdot \frac{temp - t_{opt}}{ct_{max} - t_{opt}}}}{%
rate = rmax.(ctmax - temp / ctmax - topt)^2.exp(x.(temp-topt/ctmax-topt))}
\deqn{where: x = \frac{w^{2}}{400}\cdot\bigg(1 + \sqrt{1 + \frac{40}{w}}\bigg)^{2}}{%
where x = (w^2/400).(1 + sqrt(1+(40/w))^2)}
\deqn{and:\ w = (q_{10} - 1)\cdot (ct_{max} - t_{opt})}{%
and: w = (q10 - 1)*(ctmax - topt)}

Start values in \code{get_start_vals} are derived from the data and previous values in the literature

Limits in \code{get_lower_lims} and \code{get_upper_lims} are based on extreme values that are unlikely to occur in ecological settings.
}
\note{
Generally we found this model easy to fit.
}
\examples{
# load in ggplot
library(ggplot2)

# subset for the first TPC curve
data('chlorella_tpc')
d <- subset(chlorella_tpc, curve_id == 1)

# get start values and fit model
start_vals <- get_start_vals(d$temp, d$rate, model_name = 'oneill_1972')
# fit model
mod <- nls.multstart::nls_multstart(rate~oneill_1972(temp = temp, rmax, ctmax, topt, q10),
data = d,
iter = c(4,4,4,4),
start_lower = start_vals - 10,
start_upper = start_vals + 10,
lower = get_lower_lims(d$temp, d$rate, model_name = 'oneill_1972'),
upper = get_upper_lims(d$temp, d$rate, model_name = 'oneill_1972'),
supp_errors = 'Y',
convergence_count = FALSE)

# look at model fit
summary(mod)

# get predictions
preds <- data.frame(temp = seq(min(d$temp), max(d$temp), length.out = 100))
preds <- broom::augment(mod, newdata = preds)

# plot
ggplot(preds) +
geom_point(aes(temp, rate), d) +
geom_line(aes(temp, .fitted), col = 'blue') +
theme_bw()

}
\references{
O’Neill, R.V., Goldstein, R.A., Shugart, H.H., Mankin, J.B. Terrestrial Ecosystem Energy Model. Eastern Deciduous Forest Biome Memo Report Oak Ridge. The Environmental Sciences Division of the Oak Ridge National Laboratory. (1972)
}
