# section A construct randomized versions of GOGOA3

#' randomGODB
#' 
#' @import GO.db
#' @import minimalistGODB
#' @import graphics
#' @import stats
#'
#' @description driver to construct a randomized version of GOGOA3
#' 
#' @param GOGOA3 return value of minimalistGODB::buildGODatabase()
#' @param verbose Boolean if TRUE print out some information
#' 
#' @details
#' The results of characterizeDB() show that a gene mapping to a leaf node maps
#'  to only around 10% of the ancestors. So I do not need to use a more sophisticated
#'  method to generate a random database. That is, I do not need to maintain a
#'  consistency between leaf and ancestor mappings. Therefore a very simple randomization
#'  - simply scrambling the genes in an ontology of GOGOA3 will suffice.
#'
#' @examples
#' \dontrun{
#' # GOGOA3.RData is too large to include in the R package
#' # so I need to load it from a file that is not in the package.
#' # Since this is in a file in my own file system, I could not
#' # include this as a regular example in the package.
#' # This example is given in full detail in the package vignette.
#' # You can generate GOGOA3.RData using the package 'minimalistGODB'
#' # or you can retrieve it from https://github.com/barryzee/GO/databases
#' dir<-"/Users/barryzeeberg/personal/GODB_RDATA/goa_human/"
#' load(sprintf("%s/%s",dir,"GOGOA3_goa_human.RData"))
#' GOGOA3R<-randomGODB(GOGOA3)
#' }
#' 
#' @return description
#'
#' @export
randomGODB<-
  function(GOGOA3,verbose=TRUE) {
    GOGOA3R<-GOGOA3
    DBstats(GOGOA3,"original",verbose)
    ontologies<-names(GOGOA3$ontologies)
    for(ontology in ontologies) {
      t1<-sort(table(GOGOA3$ontologies[[ontology]][,"HGNC"]),decreasing=TRUE)
      GOGOA3R$ontologies[[ontology]][,"HGNC"]<-sample(GOGOA3$ontologies[[ontology]][,"HGNC"])
      t2<-sort(table(GOGOA3R$ontologies[[ontology]][,"HGNC"]),decreasing=TRUE)
      if(!identical(t1,t2))
        stop("!identical(t1,t2)")
    }
    DBstats(GOGOA3R,"randomized",verbose)
    compare2DB(GOGOA3,GOGOA3R,verbose)
    return(GOGOA3R)
  }

#' postProcess
#' 
#' @description adds secondary components to database like GOGOA3$genes etc
#' 
#' @param l return value of randomGODB2()
#'
#' @examples
#' \dontrun{
#' # GOGOA3.RData is too large to include in the R package
#' # so I need to load it from a file that is not in the package.
#' # Since this is in a file in my own file system, I could not
#' # include this as a regular example in the package.
#' # This example is given in full detail in the package vignette.
#' # You can generate GOGOA3.RData using the package 'minimalistGODB'
#' # or you can retrieve it from https://github.com/barryzee/GO/databases
#' dir<-"/Users/barryzeeberg/personal/GODB_RDATA/goa_human/"
#' load(sprintf("%s/%s",dir,"GOGOA3_goa_human.RData"))
#' pp<-postProcess(randomGODB(GOGOA3))
#' }
#' 
#' @return returns a database like GOGOA3
#' 
#' @export
postProcess<-
  function(l) {
    l$cats<-list()
    l$genes<-list()
    l$stats<-list()
    l$stats$ncats<-list()
    l$stats$ngenes<-list()
    ontologies<-names(l$ontologies)
    for(ONTOLOGY in ontologies) {
      l$cats[[ONTOLOGY]]<-unique(l$ontologies[[ONTOLOGY]][,"GO_NAME"])
      l$stats$ncats[[ONTOLOGY]]<-length(l$cats[[ONTOLOGY]])
      l$genes[[ONTOLOGY]]<-unique(l$ontologies[[ONTOLOGY]][,"HGNC"])
      l$stats$ngenes[[ONTOLOGY]]<-length(l$genes[[ONTOLOGY]])
      
      ngenes<-as.matrix(sort(table(l$ontologies[[ONTOLOGY]][,"GO_NAME"]),decreasing=TRUE))
      fgenes<-ngenes/l$stats$ngenes[[ONTOLOGY]]
      l$stats$tcats[[ONTOLOGY]]<-cbind(ngenes,fgenes)
      
      l$genes[["ALL"]]<-unique(c(l$genes[["ALL"]],l$genes[[ONTOLOGY]]))
      l$cats[["ALL"]]<-unique(c(l$cats[["ALL"]],l$cats[[ONTOLOGY]]))
      
    }
    
    return(l)
  }

# section B characterization of databases

#' DBstats
#'
#' @description display some gene and category stats
#' 
#' @param DB GOGOA3 or a randomized version of it
#' @param ontology character c("biological_process","molecular_function","cellular_component")
#' @param title character if not null, title for output
#' @param verbose Boolean if TRUE print out some information
#'
#' @examples
#' \dontrun{
#' # GOGOA3.RData is too large to include in the R package
#' # so I need to load it from a file that is not in the package.
#' # Since this is in a file in my own file system, I could not
#' # include this as a regular example in the package.
#' # This example is given in full detail in the package vignette.
#' # You can generate GOGOA3.RData using the package 'minimalistGODB'
#' # or you can retrieve it from https://github.com/barryzee/GO/databases
#' dir<-"/Users/barryzeeberg/personal/GODB_RDATA/goa_human/"
#' load(sprintf("%s/%s",dir,"GOGOA3_goa_human.RData"))
#' DBstats(GOGOA3,ontology="biological_process")
#' }
#' 
#' @return returns no values, but prints out some stats
#' 
#' @export
DBstats<-
  function(DB,title=NULL,ontology="biological_process",verbose=TRUE) {
    if((!is.null(title))) {
      if(verbose) {
        print("",quote=FALSE)
        print("",quote=FALSE)
        print(title,quote=FALSE)
        print("",quote=FALSE)
      }
    }
    if(verbose) {
      print(sprintf("Dimensions of the %s ontology:",ontology),quote=FALSE)
      print(dim(DB$ontologies[[ontology]]),quote=FALSE)
    
      print("",quote=FALSE)
    
      print(sprintf("A few lines from the %s ontology:",ontology),quote=FALSE)
      print(DB$ontologies[[ontology]][1:3,],quote=FALSE)
    
      print("",quote=FALSE)
    
      print(sprintf("Highest hitter genes and mean from the %s ontology:",ontology),quote=FALSE)
    }
    
    t<-sort(table(DB$ontologies[[ontology]][,"HGNC"]),decreasing=TRUE)
    
    if(verbose) {
      print(t[1:5],quote=FALSE)
      print(mean(t),quote=FALSE)
    
      print("",quote=FALSE)
    
      print(sprintf("Highest hitter categories and mean from the %s ontology:",ontology),quote=FALSE)
    }
    t<-sort(table(DB$ontologies[[ontology]][,"GO"]),decreasing=TRUE)
    if(verbose) {
      print(t[1:5],quote=FALSE)
      print(mean(t),quote=FALSE)
    }
  }

#' sizeGOcats
#' 
#' @description characterize size of GO categories
#' 
#' @param GOGOA3 return value of minimalistGODB::buildGODatabase()
#' @param ontology character c("biological_process","molecular_function","cellular_component")
#' @param verbose Boolean if TRUE print out some information
#' 
#' @examples
#' \dontrun{
#' # GOGOA3.RData is too large to include in the R package
#' # so I need to load it from a file that is not in the package.
#' # Since this is in a file in my own file system, I could not
#' # include this as a regular example in the package.
#' # This example is given in full detail in the package vignette.
#' # You can generate GOGOA3.RData using the package 'minimalistGODB'
#' # or you can retrieve it from https://github.com/barryzee/GO/databases
#' dir<-"/Users/barryzeeberg/personal/GODB_RDATA/goa_human/"
#' load(sprintf("%s/%s",dir,"GOGOA3_goa_human.RData"))
#' sizeGOcats(GOGOA3,ontology="biological_process")
#' }
#' 
#' @return returns no values, but has side effect of printing out information
#' 
#' @export
sizeGOcats<-
  function(GOGOA3,ontology,verbose=TRUE) {
    t<-sort(table(GOGOA3$ontologies[[ontology]][,"GO"]),decreasing=TRUE)
    hist(log10(t),xlab="log10(Number of Gene Mappings)",
         ylab="Fraction of GO",
         main="Fraction of GO by Number of Gene Mappings")
    
    ECDF<-ecdf(t)
    plot(ECDF,
         xlab="Number of Gene Mappings",
         ylab="Fraction of GO",
         main="Fraction of GO by Number of Gene Mappings")
    plot(ECDF,xlim=c(0,10),
         xlab="Number of Gene Mappings",
         ylab="Fraction of GO",
         main="Fraction of GO by Number of Gene Mappings")
    
    tt<-table(t)
    if(verbose) {
      print("number of tiny GO categories:",quote=FALSE)
      print(tt[1:10],quote=FALSE)
    }
  }

# section C determine ancestor mappings of database leaf nodes

# Because of the hierarchical nature, one might hypothesize that
# a gene mapping to a leaf GO node is propagated upward through
# all of its parents and ancestors. To test this hypothesis, I can
# pick an arbitrary gene and see which leaf nodes it maps to. Then
# count how frequently that gene is mapped to ancestors of that leaf
# node. It turns out that the hypothesis is not true, since only around
# 10% or less of the ancestors contain a mapping to the gene in the leaf node
# The results of characterizeDB() show that a gene mapping to a leaf node maps
# to only around 10% of the ancestors. So I do not need to use a more sophisticated
# method to generate a random database. That is, I do not need to maintain a
# consistency between leaf and ancestor mappings. Therefore a very simple randomization
# - using randomGODB() simply scrambling the genes in GOGOA3$ontologies[[ontology]][,"HGNC"]
#  will suffice.

#' characterizeDB
#'
#' @description compute distribution of GO category sizes,
#'  and fraction of a leaf's ancestors containing a bait gene
#' 
#' @param GOGOA3 return value of minimalistGODB::buildGODatabase()
#' @param ontology character c("biological_process","molecular_function","cellular_component")
#' @param ngene integer number of genes to examine within range of 'hitters'
#' @param GOBPCHILDREN GO.db data set 
#' @param GOBPANCESTOR GO.db data set
#' @param hitters character c("big","mid","lo","all") designate which portion of gene table to look at
#' @param verbose BOOLEAN if TRUE print out some information
#' 
#' @examples
#' \dontrun{
#' # GOGOA3.RData is too large to include in the R package
#' # so I need to load it from a file that is not in the package.
#' # Since this is in a file in my own file system, I could not
#' # include this as a regular example in the package.
#' # This example is given in full detail in the package vignette.
#' # You can generate GOGOA3.RData using the package 'minimalistGODB'
#' # or you can retrieve it from https://github.com/barryzee/GO/databases
#' dir<-"/Users/barryzeeberg/personal/GODB_RDATA/goa_human/"
#' load(sprintf("%s/%s",dir,"GOGOA3_goa_human.RData"))
#' ontology<-"biological_process"
#' t<-characterizeDB(GOGOA3,ontology,ngene=3,GO.db::GOBPCHILDREN,GO.db::GOBPANCESTOR,hitters="all")
#' }
#' 
#' @return returns the sorted number of GO category sizes, and also has side effect of printing out some information
#' 
#' @export
characterizeDB<-
  function(GOGOA3,ontology="biological_process",ngene=2,GOBPCHILDREN,GOBPANCESTOR,hitters="all",verbose=TRUE) {
    
    # characterize size of GO categories
    sizeGOcats(GOGOA3,ontology,verbose)
    
    # characterize number of mappings per gene
    mapPerGene(GOGOA3,ontology="biological_process",verbose)
    
    # fraction of ancestor categories to which a leaf gene maps
    hitters(GOGOA3,ontology,hitters,ngene,GOBPCHILDREN,GOBPANCESTOR,verbose)
    
    return(t)
  }

#' mapPerGene
#' 
#' @description characterize number of mappings per gene
#'
#' @param GOGOA3 return value of minimalistGODB::buildGODatabase()
#' @param ontology character c("biological_process","molecular_function","cellular_component")
#' @param verbose Boolean if TRUE print out some information 
#' 
#' @examples
#' \dontrun{
#' # GOGOA3.RData is too large to include in the R package
#' # so I need to load it from a file that is not in the package.
#' # Since this is in a file in my own file system, I could not
#' # include this as a regular example in the package.
#' # This example is given in full detail in the package vignette.
#' # You can generate GOGOA3.RData using the package 'minimalistGODB'
#' # or you can retrieve it from https://github.com/barryzee/GO/databases
#' dir<-"/Users/barryzeeberg/personal/GODB_RDATA/goa_human/"
#' load(sprintf("%s/%s",dir,"GOGOA3_goa_human.RData"))
#' mapPerGene(GOGOA3,ontology="biological_process")
#' }
#' 
#' @return returns no values, but has side effect of printing out information
#'  
#' @export
mapPerGene<-
  function(GOGOA3,ontology,verbose=TRUE) {
    if(verbose) {
      print("",quote=FALSE)
      print("",quote=FALSE)
      print("****************************************",quote=FALSE)
      print("characterize number of mappings per gene:",quote=FALSE)
    }
    
    t<-sort(table(GOGOA3$ontologies[[ontology]][,"HGNC"]),decreasing=TRUE)
    if(verbose) {
      print(c("mean:" ,mean(t)),quote=FALSE)
      print(c("median:" ,median(t)),quote=FALSE)
      print(c("variance:" ,var(t)),quote=FALSE)
    
      print("",quote=FALSE)
      print("",quote=FALSE)
    }
    
    hist(log10(t),
         xlab="log10(Number of GO Mappings)",
         ylab="Fraction of Genes",
         main="Fraction of Genes by Number of GO Mappings")
    
    ECDF<-ecdf(t)
    plot(ECDF,
         xlab="Number of GO Mappings",
         ylab="Fraction of Genes",
         main="Fraction of Genes by Number of GO Mappings")
    
    plot(ECDF,xlim=c(0,10),
         xlab="Number of GO Mappings",
         ylab="Fraction of Genes",
         main="Fraction of Genes by Number of GO Mappings")
    
    if(verbose) {
      print("genes with high number of mappings:",quote=FALSE)
      print(t[1:10])
    }
    
    tt<-table(t)
    if(verbose) {
      print("",quote=FALSE)
      print("",quote=FALSE)
      print("number of genes with tiny to mid number of mappings:",quote=FALSE)
      print(tt[1:10])
    
      print("****************************************",quote=FALSE)
      print("",quote=FALSE)
      print("",quote=FALSE)
    }
  }

#' hitters
#' 
#' @description pick genes of a size range and submit to fractAncest()
#' 
#' @param GOGOA3 return value of minimalistGODB::buildGODatabase()
#' @param ontology character c("biological_process","molecular_function","cellular_component")
#' @param hitters character c("big","mid","lo","all") designate which portion of gene table to look at
#' @param ngene integer number of genes to examine within range of 'hitters'
#' @param GOBPCHILDREN GO.db data set 
#' @param GOBPANCESTOR GO.db data set
#' @param verbose Boolean if TRUE print out some information
#' 
#' @examples
#' \dontrun{
#' # GOGOA3.RData is too large to include in the R package
#' # so I need to load it from a file that is not in the package.
#' # Since this is in a file in my own file system, I could not
#' # include this as a regular example in the package.
#' # This example is given in full detail in the package vignette.
#' # You can generate GOGOA3.RData using the package 'minimalistGODB'
#' # or you can retrieve it from https://github.com/barryzee/GO/databases
#' dir<-"/Users/barryzeeberg/personal/GODB_RDATA/goa_human/"
#' load(sprintf("%s/%s",dir,"GOGOA3_goa_human.RData"))
#' hitters(GOGOA3,ontology="biological_process",hitters="all",
#'  5,GO.db::GOBPCHILDREN,GO.db::GOBPANCESTOR)
#' }
#' 
#' @return returns no values, but has side effect of printing out some information
#' 
#' @export
hitters<-
  function(GOGOA3,ontology,hitters,ngene,GOBPCHILDREN,GOBPANCESTOR,verbose=TRUE) {
    if(verbose) {
      print("",quote=FALSE)
      print("",quote=FALSE)
      print("****************************************",quote=FALSE)
      print("a few examples of specific gene names with a certain range of mappings:",quote=FALSE)
      print("",quote=FALSE)
    }
    
    t<-sort(table(GOGOA3$ontologies[[ontology]][,"HGNC"]),decreasing=TRUE)
    
    if(verbose) {
      print("hitters")
      print(t[1:10])
    }
    
    lt<-length(t)
    if(hitters=="big" | hitters=="all") {
      # pick some genes that have a lot of mappings
      goodGenes<-names(t[1:ngene])
      if(verbose) {
        print(c("BIG HITTERS",t[1:ngene]),quote=FALSE)
        print("",quote=FALSE)
      }
      
      #analysis of fraction of ancestor categories to which a leaf gene maps
      fractAncest(goodGenes,GOGOA3,ontology,GOBPCHILDREN,GOBPANCESTOR,verbose)
    }
    if(hitters=="mid" | hitters=="all") {
      lo<-lt/4
      hi<-lo+ngene-1
      goodGenes<-names(t[lo:hi])
      if(verbose) {
        print(c("MID HITTERS",t[lo:hi]),quote=FALSE)
        print("",quote=FALSE)
      }
      
      #analysis of fraction of ancestor categories to which a leaf gene maps
      fractAncest(goodGenes,GOGOA3,ontology,GOBPCHILDREN,GOBPANCESTOR,verbose)
    }
    if(hitters=="lo" | hitters=="all") {
      lo<-lt/2
      hi<-lo+ngene-1
      goodGenes<-names(t[lo:hi])
      if(verbose) {
        print(c("LOW HITTERS",t[lo:hi]),quote=FALSE)
        print("",quote=FALSE)
      }
      
      #analysis of fraction of ancestor categories to which a leaf gene maps
      fractAncest(goodGenes,GOGOA3,ontology,GOBPCHILDREN,GOBPANCESTOR,verbose)
    }
    
    if(verbose) {
      print("",quote=FALSE)
      print("",quote=FALSE)
    }
  }

#' fractAncest
#' 
#' @description analysis of fraction of ancestor categories to which a leaf gene maps
#'
#' @param genes character vector list of gene names
#' @param GOGOA3 return value of minimalistGODB::buildGODatabase()
#' @param ontology character c("biological_process","molecular_function","cellular_component")
#' @param GOBPCHILDREN GO.db data set 
#' @param GOBPANCESTOR GO.db data set
#' @param verbose Boolean if TRUE print out some information
#' 
#' @examples
#' \dontrun{
#' # GOGOA3.RData is too large to include in the R package
#' # so I need to load it from a file that is not in the package.
#' # Since this is in a file in my own file system, I could not
#' # include this as a regular example in the package.
#' # This example is given in full detail in the package vignette.
#' # You can generate GOGOA3.RData using the package 'minimalistGODB'
#' # or you can retrieve it from https://github.com/barryzee/GO/databases
#' dir<-"/Users/barryzeeberg/personal/GODB_RDATA/goa_human/"
#' load(sprintf("%s/%s",dir,"GOGOA3_goa_human.RData"))
#' ontology<-"biological_process"
#' genes<-c("CDC45","CELF2")
#' fractAncest(genes,GOGOA3,ontology,GO.db::GOBPCHILDREN,GO.db::GOBPANCESTOR)
#' }
#'
#' @return returns no values, but has side effect of printing out some results
#'  
#' @export
fractAncest<-
  function(genes,GOGOA3,ontology="biological_process",GOBPCHILDREN,GOBPANCESTOR,verbose=TRUE) {
    if(verbose) {
      print("",quote=FALSE)
      print("",quote=FALSE)
      print("****************************************",quote=FALSE)
      print("analysis of fraction of ancestor categories to which a leaf gene maps:",quote=FALSE)
    }
    
    for(gene in genes) {
      if(verbose) {
        print("",quote=FALSE)
        print("......................",quote=FALSE)
        print(c("GENE",gene))
      }
      w<-which(GOGOA3$ontologies[[ontology]][,"HGNC"]==gene)
      goodCats<-unique(GOGOA3$ontologies[[ontology]][w,"GO"])
      if(verbose)
        print(c("goodCats",length(goodCats)),quote=FALSE)
      
      # which of those cats are leaf nodes?
      BP_LEAF<-leafList(GOBPCHILDREN)
      goodLeafs<-intersect(goodCats,names(BP_LEAF))
      if(verbose)
        print(c("goodLeafs",length(goodLeafs)),quote=FALSE)
      
      BP_ANCESTOR<-addName2List(GOBPANCESTOR)
      for(leaf in goodLeafs) {
        if(verbose)
          print(c("LEAF",leaf),quote=FALSE)
        ancestors<-BP_ANCESTOR[[leaf]]
        if(verbose)
          print(c("ancestors",length(ancestors)),quote=FALSE)
        
        goodAncestors<-intersect(goodCats,ancestors)
        if(verbose)
          print(c("goodAncestors",length(goodAncestors),round(length(goodAncestors)/length(ancestors),3)),quote=FALSE)
      }
    }
    if(verbose) {
      print("****************************************",quote=FALSE)
      print("",quote=FALSE)
      print("",quote=FALSE)
    }
  }

#' leafList
#'
#' @description retrieve leaf nodes
#' 
#' @param GOBPCHILDREN GO.db dataset
#' 
#' @examples
#' BP_LEAF<-leafList(GO.db::GOBPCHILDREN)
#' 
#' @return returns a list of leaf nodes
#' 
#' @export
leafList<-
  function(GOBPCHILDREN) {
    BP_CHILDREN<-as.list(GOBPCHILDREN)
    BP_LEAF<-BP_CHILDREN[is.na(BP_CHILDREN)]
    
    return(BP_LEAF)
  }

#' addName2List
#' 
#' @description add the leaf category name to the list of ancestor categories
#' 
#' @param GOBPANCESTOR GO.db data set
#' 
#' @examples
#' BP_ANCESTOR<-addName2List(as.list(GO.db::GOBPANCESTOR))
#' 
#' @return returns an augmented list of ancestor categories
#' 
#' @export
addName2List<-
  function(GOBPANCESTOR) {
    BP_ANCESTOR<-as.list(GOBPANCESTOR)
    BP_ANCESTOR2<-BP_ANCESTOR
    for(i in 1:length(BP_ANCESTOR))
      BP_ANCESTOR2[[i]]<-c(names(BP_ANCESTOR[i]),BP_ANCESTOR[[i]])
    return(BP_ANCESTOR2)
  }

#' compare2DB
#' 
#' @description compare pairs of GO_HGNC in 2 databases
#' 
#' @param GOGOA3 return value of minimalistGODB::buildGODatabase()
#' @param GOGOA3R a supposedly randomized version of GOGOA3
#' @param verbose Boolean if TRUE print out some information
#' 
#' @examples
#' \dontrun{
#' # GOGOA3.RData is too large to include in the R package
#' # so I need to load it from a file that is not in the package.
#' # Since this is in a file in my own file system, I could not
#' # include this as a regular example in the package.
#' # This example is given in full detail in the package vignette.
#' # You can generate GOGOA3.RData using the package 'minimalistGODB'
#' # or you can retrieve it from https://github.com/barryzee/GO/databases
#' dir<-"/Users/barryzeeberg/personal/GODB_RDATA/goa_human/"
#' load(sprintf("%s/%s",dir,"GOGOA3_goa_human.RData"))
#' GOGOA3R<-randomGODB(GOGOA3)
#' compare2DB(GOGOA3,GOGOA3R)
#' }
#' 
#' @return returns no values, but has side effect of printing information
#' 
#' @export
compare2DB<-
  function(GOGOA3,GOGOA3R,verbose=TRUE) {
    ontologies<-names(GOGOA3$ontologies)
    for(ontology in ontologies) {
      GO<-GOGOA3$ontologies[[ontology]][,"GO"]
      HGNC<-GOGOA3$ontologies[[ontology]][,"HGNC"]
      pairs1<-sprintf("%s_%s",GO,HGNC)
      
      GO<-GOGOA3R$ontologies[[ontology]][,"GO"]
      HGNC<-GOGOA3R$ontologies[[ontology]][,"HGNC"]
      pairs2<-sprintf("%s_%s",GO,HGNC)
      
      int1<-unique(intersect(pairs1,pairs1))
      int2<-unique(intersect(pairs2,pairs2))
      int12<-unique(intersect(pairs1,pairs2))
      if(verbose)
        print(c(ontology,length(int1),length(int2),length(int12)),quote=FALSE)
    }
  }

#' geneListDistHitters
#' 
#' @description compute number of GOGOA3 mappings for genes in geneList
#'  
#' @param geneList character vector listg of gene names
#' @param GOGOA3 return value of minimalistGODB::buildGODatabase()
#' @param ontologies character c("biological_process","molecular_function","cellular_component")
#' @param verbose Boolean if TRUE print out some information 
#' 
#' @examples
#' \dontrun{
#' # GOGOA3.RData is too large to include in the R package
#' # so I need to load it from a file that is not in the package.
#' # Since this is in a file in my own file system, I could not
#' # include this as a regular example in the package.
#' # This example is given in full detail in the package vignette.
#' # You can generate GOGOA3.RData using the package 'minimalistGODB'
#' # or you can retrieve it from https://github.com/barryzee/GO/databases
#' dir<-"/Users/barryzeeberg/personal/GODB_RDATA/goa_human/"
#' load(sprintf("%s/%s",dir,"GOGOA3_goa_human.RData"))
#' geneList<-GOGOA3$ontologies[["biological_process"]][1:10,"HGNC"]
#' geneListDistHitters(geneList,GOGOA3)
#' }
#' 
#' @return returns no value, but has side effect of printing information
#' 
#' @export
geneListDistHitters<-
  function(geneList,GOGOA3,ontologies=NULL,verbose=TRUE) {
    if(is.null(ontologies))
      ontologies<-names(GOGOA3$ontologies)
    for(ontology in ontologies) {
      t<-sort(table(GOGOA3$ontologies[[ontology]][,"HGNC"]),decreasing=TRUE)
      if(verbose) {
        print(ontology,quote=FALSE)
        print(sort(t[geneList],decreasing=TRUE))
      }
    }
  }
