% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_cie_model.R
\name{fit_cie_model}
\alias{fit_cie_model}
\title{Fit CIE sky model}
\usage{
fit_cie_model(
  rr,
  sun_angles,
  custom_sky_coef = NULL,
  std_sky_no = NULL,
  general_sky_type = NULL,
  method = c("Nelder-Mead", "BFGS", "CG", "SANN")
)
}
\arguments{
\item{rr}{list, typically the output of \code{\link[=extract_rr]{extract_rr()}}.
If generated by other means, it must contain:
\describe{
\item{\code{zenith_dn}}{numeric vector of length one.}
\item{\code{sky_points}}{\code{data.frame} with columns \code{a} (azimuth, deg),
\code{z} (zenith, deg), and \code{rr} (relative radiance).}
}}

\item{sun_angles}{named numeric vector of length two, with components
\code{z} and \code{a} in degrees, e.g., \code{c(z = 49.3, a = 123.1)}.
See \code{\link[=estimate_sun_angles]{estimate_sun_angles()}} for details.}

\item{custom_sky_coef}{numeric vector of length five, or numeric matrix
with five columns. Custom starting coefficients for optimization.
If not provided, coefficients are initialized from standard skies.}

\item{std_sky_no}{numeric vector. Standard sky numbers as in \link{cie_table}.
If not provided, all are used.}

\item{general_sky_type}{character vector of length one. Must be \code{"Overcast"},
\code{"Clear"}, or \code{"Partly cloudy"}. See column \code{general_sky_type} in
\link{cie_table} for details. If not provided, all sky types are used.}

\item{method}{character vector. Optimization methods passed to
\code{\link[stats:optim]{stats::optim()}}. See that function for supported names.}
}
\value{
List with the following components:
\describe{
\item{\code{rr}}{The input \code{rr} with an added \code{pred} column in
\code{sky_points}, containing predicted values.}
\item{\code{opt_result}}{List returned by \code{\link[stats:optim]{stats::optim()}}.}
\item{\code{coef}}{Numeric vector of length five. CIE model coefficients.}
\item{\code{sun_angles}}{Numeric vector of length two. Sun zenith and azimuth
(degrees).}
\item{\code{method}}{Character string. Optimization method used.}
\item{\code{start}}{Numeric vector of length five. Starting parameters.}
\item{\code{metric}}{Numeric value. Mean squared deviation as in
\insertCite{Gauch2003;textual}{rcaiman}.}
}
}
\description{
Fit the CIE sky model to data sampled from a canopy photograph using
general-purpose optimization.
}
\details{
The method is based on \insertCite{Lang2010;textual}{rcaiman}.
For best results, the input data should show a linear relation between
digital numbers and the amount of light reaching the sensor. See
\code{\link[=extract_radiometry]{extract_radiometry()}} and \code{\link[=read_caim_raw]{read_caim_raw()}} for details.
As a compromise solution, \code{\link[=invert_gamma_correction]{invert_gamma_correction()}} can be used.
}
\section{Background}{

This function is based on \insertCite{Lang2010;textual}{rcaiman}. In theory,
the best result would be obtained with data showing a linear relation between
digital numbers and the amount of light reaching the sensor. See
\code{\link[=extract_radiometry]{extract_radiometry()}} and \code{\link[=read_caim_raw]{read_caim_raw()}} for further details. As a
compromise solution, \code{\link[=invert_gamma_correction]{invert_gamma_correction()}} can be used.
}

\section{Digitizing sky points with ImageJ}{

The \href{https://imagej.net/ij/docs/guide/146-19.html#sec:Multi-point-Tool}{point selection tool of ‘ImageJ’ software}
can be used to manually digitize points and create a CSV file from which to
read coordinates (see \emph{Examples}). After digitizing the points on the image,
this is a recommended workflow: 1. Use the dropdown menu \emph{Analyze > Measure}
to open the Results window. 2. Use \emph{File > Save As...} to obtain the CSV
file.

Use this code to create the input \code{sky_points} from ImageJ data:

\if{html}{\out{<div class="sourceCode">}}\preformatted{sky_points <- read.csv(path)
sky_points <- sky_points[c("Y", "X")]
colnames(sky_points) <- c("row", "col")
}\if{html}{\out{</div>}}
}

\section{Digitizing sky points with QGIS}{

To use the \href{https://qgis.org/}{QGIS software} to manually digitize
points, drag and drop the image in an empty project,
create an new vector layer, digitize points manually, save the editions, and
close the project.

To create the new vector layer, this is a recommended workflow:
\enumerate{
\item Fo to the dropdown menu \emph{Layer > Create Layer > New Geopackage Layer...}
\item Choose "point" in the Geometry type dropdown list
\item Make sure the CRS is EPSG:7589.
\item Click on the \emph{Toogle Editing} icon
\item Click on the \emph{Add Points Feature} icon.
}

Use this code to create the input \code{sky_points} from QGIS data:

\if{html}{\out{<div class="sourceCode">}}\preformatted{sky_points <- terra::vect(path)
sky_points <- terra::extract(caim, sky_points, cells = TRUE)
sky_points <- terra::rowColFromCell(caim, sky_points$cell) \%>\% as.data.frame()
colnames(sky_points) <- c("row", "col")
}\if{html}{\out{</div>}}
}

\examples{
\dontrun{
caim <- read_caim()
z <- zenith_image(ncol(caim), lens())
a <- azimuth_image(z)

# Manual method following Lang et al. (2010)
path <- system.file("external/sky_points.csv",
                    package = "rcaiman")
sky_points <- read.csv(path)
sky_points <- sky_points[c("Y", "X")]
colnames(sky_points) <- c("row", "col")
head(sky_points)
plot(caim$Blue)
points(sky_points$col, nrow(caim) - sky_points$row, col = 2, pch = 10)

# x11()
# plot(caim$Blue)
# sun_angles <- click(c(z, a), 1) \%>\% as.numeric()
sun_angles <- c(z = 49.5, a = 27.4) #taken with above lines then hardcoded

sun_row_col <- row_col_from_zenith_azimuth(z, a,
                                           sun_angles["z"],
                                           sun_angles["a"])
points(sun_row_col[2], nrow(caim) - sun_row_col[1],
       col = "yellow", pch = 8, cex = 3)

rr <- extract_rr(caim$Blue, z, a, sky_points)

set.seed(7)
model <- fit_cie_model(rr, sun_angles,
                           general_sky_type = "Clear")

plot(model$rr$sky_points$rr, model$rr$sky_points$pred)
abline(0,1)
lm(model$rr$sky_points$pred~model$rr$sky_points$rr) \%>\% summary()

sky <- cie_image(z, a, model$sun_angles, model$coef) * model$rr$zenith_dn

plot(sky)
ratio <- caim$Blue/sky
plot(ratio)
plot(ratio > 1.05)
plot(ratio > 1.15)
}
}
\references{
\insertAllCited{}
}
