#' Get filter details and names.
#'
#' @export
#' @name filters
#' @param x (character) Optional filter name. If not given, all filters
#' returned.
#' @details Note that all filter names in this package have periods 
#' and dashes replaced with underscores as they both cause problems 
#' in an R console.
#' @examples
#' filter_names()
#' filter_details()
#' filter_details()$has_authenticated_orcid
#' filter_details()$has_authenticated_orcid$possible_values
#' filter_details()$has_authenticated_orcid$description
#' filter_details("issn")
#' filter_details("iss")
#' filter_details(c("issn", "alternative_id"))
filter_names <- function() filter_list

#' @export
#' @rdname filters
filter_details <- function(x = NULL) {
  if (is.null(x)) {
    filter_deets
  } else {    
    filter_deets[match.arg(x, names(filter_deets), several.ok = TRUE)]
  }
}


# helpers ----------------------------
filter_handler <- function(x){
  if (is.null(x)) {
    NULL
  } else {
    nn <- names(x)
    if (any(nn %in% other_filters)) {
      nn <- sapply(nn, function(x) {
        if (x %in% other_filters) {
          switch(
           x,
           license_url = 'license.url',
           license_version = 'license.version',
           license_delay = 'license.delay',
           full_text_version = 'full-text.version',
           full_text_type = 'full-text.type',
           full_text_application = 'full-text.application',
           award_number = 'award.number',
           award_funder = 'award.funder',
           relation_type = 'relation.type',
           relation_object = 'relation.object',
           relation_object_type = 'relation.object-type'
          )
        } else {
          x
        }
      }, USE.NAMES = FALSE)
    }
    newnn <- gsub("_", "-", nn)
    names(x) <- newnn
    x <- sapply(x, asl)
    args <- list()
    for (i in seq_along(x)) {
      args[[i]] <- paste(names(x[i]), unname(x[i]), sep = ":")
    }
    paste0(args, collapse = ",")
  }
}

asl <- function(z) {
  # z <- tolower(z)
  if (is.logical(z) || tolower(z) == "true" || tolower(z) == "false") {
    if (z) {
      return('true')
    } else {
      return('false')
    }
  } else {
    return(z)
  }
}

other_filters <- c(
  'license_url','license_version','license_delay','full_text_version',
  'full_text_type','award_number','award_funder','relation_type',
  'relation_object','relation_object_type'
)

filter_list <- c(
  'award_funder','award_number','content_domain','from_accepted_date','from_online_pub_date',
  'from_posted_date','from_print_pub_date','has_assertion','has_authenticated_orcid',
  'has_crossmark_restriction','has_relation','location','relation_object',
  'relation_object_type','relation_type','until_accepted_date',
  'until_online_pub_date','until_posted_date','until_print_pub_date',
  'has_funder','funder','prefix','member','from_index_date','until_index_date',
  'from_deposit_date','until_deposit_date','from_update_date','until_update_date',
  'from_pub_date','until_pub_date',
  'has_license','license_url','license_version','license_delay','has_full_text',
  'full_text_version','full_text_type','full_text_application',
  'has_references','reference_visibility','has_archive',
  'archive','has_orcid','orcid','issn','isbn','type','directory','doi','updates','is_update',
  'has_update_policy','container_title','category_name','type_name',
  'from_created_date', 'until_created_date', 'has_affiliation',
  'assertion_group', 'assertion', 'article_number', 'alternative_id',
  'has_clinical_trial_number', 'has_abstract', 'has_content_domain', 
  'has_domain_restriction'
)

filter_deets <- list(
  "has_funder" = list("possible_values" = NA, "description" = "metadata which includes one or more funder entry" ),
  "funder" = list("possible_values" = "{funder_id}", "description" = "metadata which include the {funder_id} in FundRef data" ),
  "prefix" = list("possible_values" = "{owner_prefix}", "description" = "metadata belonging to a DOI owner prefix {owner_prefix} (e.g. '10.1016' )" ),
  "member" = list("possible_values" = "{member_id}", "description" = "metadata belonging to a CrossRef member" ),
  "from_index_date" = list("possible_values" = '{date}', "description" = "metadata indexed since (inclusive) {date}" ),
  "until_index_date" = list("possible_values" = '{date}', "description" = "metadata indexed before (inclusive) {date}" ),
  "from_deposit_date" = list("possible_values" = '{date}', "description" = "metadata last (re)deposited since (inclusive) {date}" ),
  "until_deposit_date" = list("possible_values" = '{date}', "description" = "metadata last (re)deposited before (inclusive) {date}" ),
  "from_update_date" = list("possible_values" = '{date}', "description" = "Metadata updated since (inclusive) {date} Currently the same as 'from_deposit_date'" ),
  "until_update_date" = list("possible_values" = '{date}', "description" = "Metadata updated before (inclusive) {date} Currently the same as 'until_deposit_date'" ),
  "from_created_date" = list("possible_values" = '{date}', "description" = "metadata first deposited since (inclusive) {date}" ),
  "until_created_date" = list("possible_values" = '{date}', "description" = "metadata first deposited before (inclusive) {date}" ),
  "from_pub_date" = list("possible_values" = '{date}', "description" = "metadata where published date is since (inclusive) {date}" ),
  "until_pub_date" = list("possible_values" = '{date}', "description" = "metadata where published date is before (inclusive)  {date}" ),
  "has_license" = list("possible_values" = NA, "description" = "metadata that includes any '<license_ref>' elements" ),
  "license_url" = list("possible_values" = '{url}', "description" = "metadata where '<license_ref>' value equals {url}" ),
  "license_version" = list("possible_values" = '{string}', "description" = "metadata where the '<license_ref>''s 'applies_to' attribute  is '{string}'"),
  "license_delay" = list("possible_values" = "{integer}", "description" = "metadata where difference between publication date and the '<license_ref>''s 'start_date' attribute is <= '{integer}' (in days"),
  "has_full_text" = list("possible_values" = NA, "description" = "metadata that includes any full text '<resource>' elements_" ),
  "full_text_version" = list("possible_values" = '{string}' , "description" = "metadata where '<resource>' element's 'content_version' attribute is '{string}'" ),
  "full_text_type" = list("possible_values" = '{mime_type}' , "description" = "metadata where '<resource>' element's 'content_type' attribute is '{mime_type}' (e.g. 'application/pdf')" ),
  "full_text_application" = list("possible_values" = '{string}' , "description" = "metadata where <resource> link has one of the following intended applications: text-mining, similarity-checking or unspecified" ),
  "has_references" = list("possible_values" = NA , "description" = "metadata for works that have a list of references" ),
  "reference_visibility" = list("possible_values" = 'open, limited, closed', "description" = "metadata for works where references are either open, limited (to Metadata Plus subscribers) or closed"),
  "has_archive" = list("possible_values" = NA , "description" = "metadata which include name of archive partner" ),
  "archive" = list("possible_values" = '{string}', "description" = "metadata which where value of archive partner is '{string}'" ),
  "has_orcid" = list("possible_values" = NA, "description" = "metadata which includes one or more ORCIDs" ),
  "orcid" = list("possible_values" = '{orcid}', "description" = "metadata where '<orcid>' element's value = '{orcid}'" ),
  "issn" = list("possible_values" = '{issn}', "description" = "metadata where record has an ISSN = '{issn}' Format is 'xxxx_xxxx'." ),
  "isbn" = list("possible_values" = '{isbn}', "description" = "metadata where record has an ISBN" ),
  "type" = list("possible_values" = '{type}', "description" = "metadata records whose type = '{type}' Type must be an ID value from the list of types returned by the '/types' resource" ),
  "directory" = list("possible_values" = "{directory}", "description" = "metadata records whose article or serial are mentioned in the given '{directory}'. Currently the only supported value is 'doaj'" ),
  "doi" = list("possible_values" = '{doi}', "description" = "metadata describing the DOI '{doi}'" ),
  "updates" = list("possible_values" = '{doi}', "description" = "metadata for records that represent editorial updates to the DOI '{doi}'" ),
  "is_update" = list("possible_values" = NA, "description" = "metadata for records that represent editorial updates" ),
  "has_update_policy" = list("possible_values" = NA, "description" = "metadata for records that include a link to an editorial update policy" ),
  "container_title" = list("possible_values" = NA, "description" = "metadata for records with a publication title exactly with an exact match" ),
  "category_name" = list("possible_values" = NA, "description" = "metadata for records with an exact matching category label" ),
  "type_name" = list("possible_values" = NA, "description" = "metadata for records with an exacty matching type label" ),
  "award_number" = list("possible_values" = "{award_number}", "description" = "metadata for records with a matching award nunber_ Optionally combine with 'award_funder'" ),
  "award_funder" = list("possible_values" = '{funder DOI or id}', "description" = "metadata for records with an ffun with matching funder. Optionally combine with 'award_number'"),
  "assertion_group" = list("possible_values" = NA, "description" = "metadata for records with an assertion in a particular group" ),
  "assertion" = list("possible_values" = NA, "description" = "metadata for records with a particular named assertion" ),
  "has_affiliation" = list("possible_values" = NA, "description" = "metadata for records that have any affiliation information" ),
  "article_number" = list("possible_values" = NA, "description" = "metadata for records with a given article number"),
  "alternative_id" = list("possible_values" = NA, "description" = "metadata for records with the given alternative ID, which may be a publisher_specific ID, or any other identifier a publisher may have provided"),
  'has_clinical_trial_number' = list("possible_values" = NA, "description" = "metadata for records which include a clinical trial number"), 
  'has_abstract' = list("possible_values" = NA, "description" = "metadata for records which include an abstract"),
  "content_domain" = list("possible_values" = NA, "description" = "metadata where the publisher records a particular domain name as the location Crossmark content will appear"),
  "has_content_domain" = list("possible_values" = NA, "description" = "metadata where the publisher records a domain name location for Crossmark content"),
  "has_domain_restriction" = list("possible_values" = NA, "description" = "metadata where the publisher restricts Crossmark usage to content domains"),
  "from_accepted_date" = list("possible_values" = "{date}", "description" = "metadata where accepted date is since (inclusive) {date}"),
  "from_online_pub_date" = list("possible_values" = "{date}", "description" = "metadata where online published date is since (inclusive) {date}"),
  "from_posted_date" = list("possible_values" = "{date}", "description" = "metadata where posted date is since (inclusive) {date}"),
  "from_print_pub_date" = list("possible_values" = "{date}", "description" = "metadata where print published date is since (inclusive) {date}"),
  "has_assertion" = list("possible_values" = NA, "description" = "metadata for records with any assertions"),
  "has_authenticated_orcid" = list("possible_values" = NA, "description" = "metadata which includes one or more ORCIDs where the depositing publisher claims to have witness the ORCID owner authenticate with ORCID"),
  "has_crossmark_restriction" = list("possible_values" = NA, "description" = "metadata where the publisher restricts Crossmark usage to content domains"),
  "has_relation" = list("possible_values" = NA, "description" = "metadata for records that either assert or are the object of a relation"),
  "location" = list("possible_values" = "{country_name}", "description" = "funder records where location = {country name}. Only works on /funders route"),
  "relation_object" = list("possible_values" = NA, "description" = "Relations where the object identifier matches the identifier provided"),
  "relation_object_type" = list("possible_values" = NA, "description" = "One of the identifier types from the Crossref relations schema (e.g. doi, issn)"),
  "relation_type" = list("possible_values" = NA, "description" = "One of the relation types from the Crossref relations schema (e.g. is-referenced-by, is-parent-of, is-preprint-of)"),
  "until_accepted_date" = list("possible_values" = "{date}", "description" = "metadata where accepted date is before (inclusive) {date}"),
  "until_online_pub_date" = list("possible_values" = "{date}", "description" = "metadata where online published date is before (inclusive) {date}"),
  "until_posted_date" = list("possible_values" = "{date}", "description" = "metadata where posted date is before (inclusive) {date}"),
  "until_print_pub_date" = list("possible_values" = "{date}", "description" = "metadata where print published date is before (inclusive) {date}")
)
