
<!-- README.md is generated from README.Rmd. Please edit that file -->

# re: Regular Expressions Made Easy in R

## Overview

**re** is an R package designed to simplify working with regular
expressions by providing a set of functions similar to Python’s `re`
module. The package includes utilities for compiling regular expressions
with specific flags, checking for matches, escaping special characters,
and more. By emulating the functionality and naming conventions of
Python’s `re` module, **re** aims to make regex operations in R more
intuitive and accessible, especially for those familiar with Python.

## Installation

You can install the `re` package directly from GitHub.

``` r
# Install devtools if you haven't already
install.packages("devtools")

# Install re package from GitHub
devtools::install_github("pythonicr/re")
```

## Getting Started

### Basic Usage

Here are some examples demonstrating how to use the functions provided
by the **re** package.

#### Compile a Regular Expression with Specific Flags

The `re_compile` function compiles a regular expression pattern with
specified flags. This step is optional as flags and patterns can be
provided for any of the functions.

``` r
library(re)

# Compile a pattern with case-insensitive matching
pattern <- re_compile("^abc", IGNORECASE = TRUE)
# Compile a pattern with multi-line matching (abbreviations are based on Python's re package)
pattern <- re_compile("end$", M = TRUE)
pattern <- re_compile("end$", MULTILINE = TRUE)
# Compile a pattern with DOTALL flag
pattern <- re_compile("a.b", DOTALL = TRUE)
```

#### Check if a String Contains a Regular Expression

The `re_contains` function checks whether a specified pattern is found
within each element of a character vector.

``` r
# Check if strings contain a pattern
re_contains(pattern, "Abcdef")
#> [1] FALSE
re_contains("xyz$", "hello world xyz")
#> [1] TRUE
```

#### Escape Special Characters

The `re_escape` function escapes all special characters in a regular
expression string.

``` r
# Escape special characters in a string
escaped_pattern <- re_escape("a[bc].*d?")
print(escaped_pattern)
#> [1] "a\\[bc\\]\\.\\*d\\?"
```

#### Extract All Occurrences of a Pattern in a String

The `re_findall` function extracts all occurrences of a specified
pattern from each element of a character vector.

``` r
# Extract all words from a string
pattern <- re_compile("\\b\\w+\\b")
re_findall(pattern, "This is a test.")
#> [[1]]
#> [1] "This" "is"   "a"    "test"
re_findall("\\d+", "123 abc 456")
#> [[1]]
#> [1] "123" "456"
```

#### Match a Pattern Against the Entire String

The `re_fullmatch` function checks whether each element of a character
vector fully matches a specified pattern.

``` r
# Check for full matches in a string
pattern <- re_compile("\\d{3}-\\d{2}-\\d{4}")
re_fullmatch(pattern, "123-45-6789")
#> [[1]]
#> [1] "123-45-6789"
re_fullmatch("123-45-6789", "123-45-6789 and more")
#> [[1]]
#> [1] NA
```

#### Match a Pattern at the Start of a String

The `re_match` function checks whether each element of a character
vector matches a specified pattern at the start.

``` r
# Check for matches at the start of a string
pattern <- re_compile("\\d{3}")
re_match(pattern, "123abc")
#> [[1]]
#> [1] "123"
re_match("abc", "xyzabc")
#> [[1]]
#> [1] NA
```

#### Search for a Pattern in a String

The `re_search` function searches for occurrences of a specified pattern
within each element of a character vector.

``` r
# Search for a pattern in a string
pattern <- re_compile("\\d+")
re_search(pattern, "abc 123 xyz")
#> [[1]]
#> [1] "123"
re_search("\\bword\\b", "A sentence with the word.")
#> [[1]]
#> [1] "word"
```

#### Split a String by a Regular Expression Pattern

The `re_split` function splits each element of a character vector into
substrings based on a specified pattern.

``` r
# Split strings based on a pattern
pattern <- re_compile("\\s+")
re_split(pattern, "Split this string")
#> [[1]]
#> [1] "Split"  "this"   "string"
re_split("\\W+", "Split,with!punctuation.morestuff", maxsplit = 2)
#> [[1]]
#> [1] "Split"                 "with"                  "punctuation.morestuff"
```

#### Substitute Occurrences of a Pattern in a String

The `re_sub` function replaces all occurrences of a specified pattern in
each element of a character vector with a replacement string.

``` r
# Substitute patterns in a string
pattern <- re_compile("\\d+")
re_sub(pattern, "number", "Replace 123 with text.")
#> [1] "Replace number with text."
re_sub("\\s+", "-", "Split and join")
#> [1] "Split-and-join"
```

## Contributing

We welcome contributions to the **re** package. If you have suggestions,
bug reports, or want to contribute code, please open an issue or submit
a pull request on our GitHub repository.

- [x] compile
- [x] escape
- [x] findall
- [ ] finditer
- [x] fullmatch
- [x] match
- [ ] purge
- [x] search
- [x] split
- [x] sub
- [ ] subn

## License

**re** is released under the MIT License. See the LICENSE file in the
package’s repository for more details.
