% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AR_reml.R
\name{fitAR_map}
\alias{fitAR_map}
\title{Map-level AR REML}
\usage{
fitAR_map(
  Y,
  coords,
  formula = "y ~ t",
  X.list = list(t = 1:ncol(Y)),
  resids.only = FALSE
)
}
\arguments{
\item{Y}{a spatiotemporal response variable: a numeric matrix or data frame
where columns correspond to time points and rows correspond to pixels.}

\item{coords}{a numeric coordinate matrix or data frame, with two columns and
rows corresponding to each pixel}

\item{formula}{a model formula, passed to \code{fitAR()}: the left side
of the formula should always be "y" and the right hand side should refer to
variables in \code{X.list}}

\item{X.list}{a named list of temporal or spatiotemporal predictor variables:
elements must be either numeric vectors with one element for each time point
or a matrix/data frame with rows corresponding to pixels and columns
corresponding to time point. These elements must be named and referred to
in \code{formula}}

\item{resids.only}{logical: should output beyond coordinates and residuals be
withheld? Useful when passing output to \code{fitCor()}}
}
\value{
\code{fitCLS_map} returns a list object of class "mapTS".

The output will always contain at least these elements:

\describe{
   \item{call}{the function call}
   \item{coords}{the coordinate matrix or dataframe}
   \item{residuals}{time series residuals: rows correspond to pixels
   (\code{coords})}
}

When \code{resids.only = FALSE}, the output will also contain the following
components. Matrices have rows that correspond to pixels and columns that
correspond to time points and vector elements correspond to pixels.

\describe{
   \item{coefficients}{a numeric matrix of coefficeints}
   \item{SEs}{a numeric matrix of coefficient standard errors}
   \item{tstats}{a numeric matrix of t-statistics for coefficients}
   \item{pvals}{a numeric matrix of p-values for coefficients t-tests}
   \item{rhos}{a vector of rho values for each pixel}
   \item{MSEs}{a numeric vector of MSEs}
   \item{logLiks}{a numeric vector of log-likelihoods}
   \item{fitted.values}{a numeric matrix of fitted values}
}

An attribute called "resids.only" is also set to match the value of
\code{resids.only}
}
\description{
\code{fitAR_map} is used to fit AR REML regression to each spatial
location (pixel) within spatiotemporal data.
}
\details{
\code{fitAR_map} is a wrapper function that applies \code{fitAR} to
many pixels.

The function loops through the rows of \code{Y}, matched with rows of
spatiotemporal predictor matrices. Purely temporal predictors, given by
vectors, are used for all pixels. These predictor variables, given by the
right side of \code{formula} are sourced from named elements in \code{X.list}.
}
\examples{
# simulate dummy data
 time.points = 9 # time series length
 map.width = 5 # square map width
 coords = expand.grid(x = 1:map.width, y = 1:map.width) # coordinate matrix
 ## create empty spatiotemporal variables:
 X <- matrix(NA, nrow = nrow(coords), ncol = time.points) # response
 Z <- matrix(NA, nrow = nrow(coords), ncol = time.points) # predictor
# setup first time point:
 Z[, 1] <- .05*coords[,"x"] + .2*coords[,"y"]
 X[, 1] <- .5*Z[, 1] + rnorm(nrow(coords), 0, .05) #x at time t
 ## project through time:
 for(t in 2:time.points){
   Z[, t] <- Z[, t-1] + rnorm(map.width^2)
   X[, t] <- .2*X[, t-1] + .1*Z[, t] + .05*t + rnorm(nrow(coords), 0 , .25)
 }

# visualize dummy data (NOT RUN)
library(ggplot2);library(dplyr)
data.frame(coords, X) \%>\%
  reshape2::melt(id.vars = c("x", "y")) \%>\%
  ggplot(aes(x = x, y = y, fill = value)) +
  geom_tile() +
  facet_wrap(~variable)

# fit AR, showing all output
fitAR_map(X, coords, formula = y ~ t, resids.only = TRUE)

# fit AR with temporal and spatiotemporal predictors
(AR.map <- fitAR_map(X, coords, formula = y ~ t + Z, X.list = list(t = 1:ncol(X),
                     Z = Z), resids.only = FALSE))
## extract some values
AR.map$coefficients # coefficients
AR.map$logLik # log-likelihoods

## Methods
summary(AR.map)
residuals(AR.map)
coefficients(AR.map)

}
\seealso{
\code{\link{fitAR}} for fitting AR REML to individual time series and \code{\link{fitCLS}}
& \code{\link{fitCLS_map}} for time series analysis based on conditional least squares.

Other remoteTS: 
\code{\link{fitAR}()},
\code{\link{fitCLS_map}()},
\code{\link{fitCLS}()}
}
\concept{remoteTS}
