% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kappaIbd.R
\name{kappaIBD}
\alias{kappaIBD}
\title{IBD (kappa) coefficients}
\usage{
kappaIBD(
  x,
  ids = labels(x),
  inbredAction = 1,
  simplify = TRUE,
  acrossComps = TRUE,
  Xchrom = FALSE
)
}
\arguments{
\item{x}{A pedigree in the form of a \code{ped} object (or a list of such).}

\item{ids}{A character (or coercible to character) containing ID labels of
two or more pedigree members.}

\item{inbredAction}{An integer telling the program what to do if either of
the \code{ids} individuals are inbred. Possible values are: 0 = do nothing; 1 =
print a warning message (default); 2 = raise an error. In the first two
cases the coefficients are reported as \code{NA}.}

\item{simplify}{Simplify the output (to a numeric of length 3) if \code{ids} has
length 2. Default: TRUE.}

\item{acrossComps}{A logical indicating if pairs of individuals in different
components should be included. Default: TRUE.}

\item{Xchrom}{A logical, indicating if the autosomal (default) or
X-chromosomal kappa coefficients should be computed.}
}
\value{
If \code{ids} has length 2 and \code{simplify = TRUE}: A numeric vector of
length 3: \eqn{(\kappa_0, \kappa_1, \kappa_2)}.

Otherwise: A data frame with one row for each pair of individuals, and 5
columns. The first two columns contain the ID labels, and columns 3-5
contain the IBD coefficients.

Kappa coefficients of inbred individuals (meaning X-inbred females if
\code{Xchrom = T}) are reported as NA, unless \code{inbredAction = 2}.

The X-chromosomal \eqn{\kappa_2} is NA whenever at least one of the two
individuals is male.
}
\description{
Computes the three IBD coefficients summarising the relationship between two
non-inbred individuals. Both autosomal and X chromosomal versions are
implemented. The pedigree founders (other than the individuals in question)
are allowed to be inbred; see \code{\link[pedtools:founderInbreeding]{pedtools::founderInbreeding()}} for how to set
this up, and see Examples below.
}
\details{
For non-inbred individuals a and b, their autosomal IBD coefficients
\eqn{(\kappa_0, \kappa_1, \kappa_2)} are defined as follows: \deqn{\kappa_i =
P(\text{a and b share exactly i alleles IBD at a random autosomal locus})}

The autosomal kappa coefficients are computed from the kinship coefficients.
When a and b are both nonfounders, the following formulas hold:
\itemize{
\item \eqn{\kappa_2 = \varphi_{MM} \cdot \varphi_{FF} + \varphi_{MF} \cdot\varphi_{FM}}
\item \eqn{\kappa_1 = 4 \varphi_{ab} - 2 \kappa_2}
\item \eqn{\kappa_0 = 1 - \kappa_1 - \kappa_2}
}

Here \eqn{\varphi_{MF}} denotes the kinship coefficient between the
\strong{m}other of a and the \strong{f}ather of b, etc. If either a or b is a founder,
then \eqn{\kappa_2 = 0}, while the other two formulas remain as before.

The X-chromosomal IBD coefficients are defined similarly to the autosomal
case. Here \eqn{\kappa_2} is undefined when one or both individuals are male,
which greatly simplifies the calculations when males are involved. The
formulas are (with \eqn{\varphi_{ab}} now referring to the X-chromosomal
kinship coefficient):
\itemize{
\item Both male: \eqn{(\kappa_0, \kappa_1, \kappa_2) = (1-\varphi_{ab}, \varphi_{ab}, \text{NA})}
\item One male, one female: \eqn{(\kappa_0, \kappa_1, \kappa_2) = (1-2 \varphi_{ab},
2 \varphi_{ab}, \text{NA})}
\item Two females: Similar formulas as in the autosomal case.
}
}
\examples{
### Siblings
x = nuclearPed(2)
kappaIBD(x)

k = kappaIBD(x, 3:4)
stopifnot(identical(k, c(.25, .5, .25)))

### Quad half first cousins
x = quadHalfFirstCousins()
k = kappaIBD(x, ids = leaves(x))
stopifnot(identical(k, c(17/32, 14/32, 1/32)))

### Paternal half brothers with 100\% inbred father
# Genetically indistinguishable from an (outbred) father-son relationship
x = halfSibPed() |> setFounderInbreeding(ids = 2, value = 1)
plot(x, hatched = 4:5)

k = kappaIBD(x, 4:5)
stopifnot(identical(k, c(0, 1, 0)))

### X-chromosomal kappa
y = nuclearPed(2, sex = 2)
kappaIBD(y, Xchrom = TRUE)

}
\seealso{
\code{\link[=kinship]{kinship()}}, \code{\link[=identityCoefs]{identityCoefs()}} for coefficients allowing inbreeding,
\code{\link[=showInTriangle]{showInTriangle()}} for plotting kappa coefficients in the IBD triangle.
}
