#' Filter objects matching specified keys/tags
#'
#' Get objects matching at least one of the specified expressions from the input
#' and write them to the output. All objects matching the expressions will be
#' kept in the output, and objects referenced by them will also be added to the
#' output (unless `omit_referenced = TRUE`). Objects will be written out in the
#' order they are found in the input. Please note that the function will only
#' work correctly on history files if `omit_referenced` is `TRUE`, and it cannot
#' be used on change files.
#'
#' @param input_path A string. The path to the OSM data/history file to which
#'   the filters should be applied. Please see [file_formats] for a list of
#'   supported file formats.
#' @param filters A string. The filter expressions that should be applied to the
#'   input file. Please see the "Filter expressions" section for a description
#'   of the filter expression format.
#' @param output_path A string. The path to the file where the output should be
#'   written to. Please see [file_formats] for a list of supported file formats.
#' @param invert_match A logical. Whether to invert the sense of matching - i.e.
#'   to exclude objects with matching tags. Defaults to `FALSE`.
#' @param omit_referenced A logical. Whether to omit the nodes referenced from
#'   matching ways and members referenced from matching relations. Defaults to
#'   `FALSE`.
#' @param remove_tags A logical. Whether to remove tags from objects that are
#'   not matching the filter expression but are included to complete references
#'   (nodes in ways and members of relations). Defaults to `FALSE`. Please note
#'   that if an object is both matching the filter and used as a reference it
#'   will keep its tags.
#' @param overwrite A logical. Whether existing files should be overwritten by
#'   the output. Defaults to `FALSE`.
#' @param echo_cmd A logical. Whether to print the Osmium command generated by
#'   the function call to the screen. Defaults to `FALSE`.
#' @param echo A logical. Whether to print the standard output and error
#'   generated by the Osmium call to the screen. Defaults to `TRUE`.
#' @param spinner A logical. Whether to show a reassuring spinner while the
#'   Osmium call is being executed. Defaults to `TRUE`.
#' @param verbose A logical. Whether to display detailed information on the
#'   running command. Defaults to `FALSE`.
#' @param progress A logical. Whether to display a progress bar while running
#'   the command. Defaults to `FALSE`.
#'
#' @section Filter expressions:
#' A filter expression specifies a tag or tags that should be found in the data
#' and the type of object (node, way or relation) that should be matched.
#'
#' The object type(s) comes first, then a slash (/) and then the rest of the
#' expression. Object types are specified as 'n' (for nodes), 'w' (for ways),
#' 'r' (for relations) and 'a' (for areas - closed ways with 4 or more nodes and
#' relations with `type=multipolygon` or `type=boundary` tag). Any combination
#' of them can be used. If the object type is not specified, the expression
#' matches all object types.
#'
#' Some examples:
#'
#' - `n/amenity` - matches all nodes with the key "amenity".
#' - `nw/highway` - matches all nodes or ways with the key "highway".
#' - `/note` - matches objects of any type with the key "note".
#' - `note` - matches objects of any type with the key "note".
#' - `w/highway=primary` - matches all ways with the key "highway" and value
#' "primary".
#' - `w/highway!=primary` - matches all ways with the key "highway" and a value
#' other than "primary".
#' - `r/type=multipolygon,boundary` - matches all relations with key "type" and
#' value "multipolygon" or "boundary".
#' - `w/name,name:de=Kastanienallee,Kastanienstrasse` - matches any way with a
#' "name" or "name:de" tag with the value "Kastanienallee" or
#' "Kastanienstrasse".
#' - `n/addr:*` - matches all nodes with any key starting with "addr:"
#' - `n/name=*Paris` - matches all nodes with a name that contains the word
#' "Paris".
#' - `a/building` - matches any closed ways with 4 or more nodes or relations
#' tagged "building". Relations must also have a tag "type=multipolygon" or
#' "type=boundary".
#'
#' If there is no equal sign ("=") in the expression, only keys are matched and
#' values can be anything. If there is an equal sign ("=") in the expression,
#' the key is to the left and the value to the right. An exclamation sign ("!")
#' before the equal sign means: a tag with that key, but not the value(s) to the
#' right of the equal sign. A leading or trailing asterisk ("*") can be used for
#' substring or prefix matching, respectively. Commas (",") can be used to
#' separate several keys or values.
#'
#' All filter expressions are case-sensitive. There is no way to escape the
#' special characters such as "=", "*" and ",". You can not mix
#' comma-expressions and "*"-expressions.
#'
#' The specified filter expressions are matched in the order they are given. To
#' achieve best performance, put expressions expected to match more often first.
#'
#' Area matches (with leading "a/") do not check whether the matched object is a
#' valid (multi)polygon, they only check whether an object might possibly be
#' turned into a (multi)polygon. This is the case for all closed ways (where the
#' first and last node are the same) with 4 or more nodes and for all relations
#' that have an additional "type=multipolygon" or "type=boundary" tag.
#'
#' @return The normalized path to the output file.
#'
#' @examplesIf identical(tolower(Sys.getenv("NOT_CRAN")), "true")
#' pbf_path <- system.file("extdata/cur.osm.pbf", package = "rosmium")
#'
#' # get all amenity nodes
#' output <- tags_filter(pbf_path, "n/amenity", tempfile(fileext = ".osm.pbf"))
#' nodes <- sf::st_read(output, layer = "points", quiet = TRUE)
#' head(nodes$other_tags)
#'
#' # get all objects (nodes, ways or relations) with an addr:* tag
#' output <- tags_filter(
#'   pbf_path,
#'   "addr:*",
#'   tempfile(fileext = ".osm.pbf"),
#'   omit_referenced = TRUE
#' )
#' nodes <- sf::st_read(output, layer = "points", quiet = TRUE)
#' head(nodes$other_tags)
#'
#' # get all nodes and ways with a highway tag and all relations tagged with
#' # type=restriction plus all referenced objects
#' output <- tags_filter(
#'   pbf_path,
#'   "nw/highway r/type=restriction",
#'   tempfile(fileext = ".osm.pbf")
#' )
#' ways <- sf::st_read(output, layer = "lines", quiet = TRUE)
#' head(ways$highway)
#' relations <- sf::st_read(output, layer = "other_relations", quiet = TRUE)
#' head(relations$other_tags)
#'
#' @export
tags_filter <- function(input_path,
                        filters,
                        output_path,
                        invert_match = FALSE,
                        omit_referenced = FALSE,
                        remove_tags = FALSE,
                        overwrite = FALSE,
                        echo_cmd = FALSE,
                        echo = TRUE,
                        spinner = TRUE,
                        verbose = FALSE,
                        progress = FALSE) {
  assert_osmium_is_installed()

  checkmate::assert_file_exists(input_path)
  checkmate::assert_logical(invert_match, any.missing = FALSE, len = 1)
  checkmate::assert_logical(omit_referenced, any.missing = FALSE, len = 1)
  checkmate::assert_logical(remove_tags, any.missing = FALSE, len = 1)
  checkmate::assert_logical(overwrite, any.missing = FALSE, len = 1)
  checkmate::assert_logical(echo_cmd, any.missing = FALSE, len = 1)
  checkmate::assert_logical(echo, any.missing = FALSE, len = 1)
  checkmate::assert_logical(spinner, any.missing = FALSE, len = 1)
  checkmate::assert_logical(verbose, any.missing = FALSE, len = 1)
  checkmate::assert_logical(progress, any.missing = FALSE, len = 1)
  assert_output_path_multi_ext(output_path, overwrite)

  filters_arg <- assert_and_assign_filters(filters)
  output_arg <- paste0("--output=", output_path)
  invert_match_arg <- if (invert_match) "--invert-match" else character()
  omit_ref_arg <- if (omit_referenced) "--omit-referenced" else character()
  remove_tags_arg <- if (remove_tags) "--remove-tags" else character()
  overwrite_arg <- if (overwrite) "--overwrite" else character()
  verbose_arg <- if (verbose) "--verbose" else character()
  progress_arg <- ifelse(progress, "--progress", "--no-progress")

  args <- c(
    "tags-filter",
    input_path,
    filters_arg,
    output_arg,
    invert_match_arg,
    omit_ref_arg,
    remove_tags_arg,
    overwrite_arg,
    verbose_arg,
    progress_arg
  )

  logs <- processx::run(
    "osmium",
    args,
    echo = echo,
    spinner = spinner,
    echo_cmd = echo_cmd
  )

  return(normalizePath(output_path))
}

assert_and_assign_filters <- function(filters) {
  checkmate::assert_string(filters)

  filters_input <- unlist(strsplit(filters, " "))

  assert_filters(filters_input, .var.name = "filters")

  return(filters_input)
}

check_filters <- function(filters_input) {
  is_option <- grepl("^-", filters_input)
  contains_option <- any(is_option)

  if (contains_option) {
    return(
      paste0(
        "Must not contain option, but found at least one: ",
        paste(paste0("'", filters_input[is_option], "'"), collapse = ", ")
      )
    )
  }

  # slashes are used to specify the object type that may be affected by the
  # filter. to do this, one should specify the object type (either n, w, r or
  # a), use a slash, and then specify the key or key-value pair used in the
  # filter - e.g nw/highway means that and any node or way tagged as highway
  # should be kept. so we have to check if the characters before the slash are
  # actually n, w, r or a, otherwise osmium throws an error.

  contains_slash <- grepl("/", filters_input)

  # the regex below extracts all the text that appears before the first slash.

  text_before_slash <- sub("(^[^/]*).*", "\\1", filters_input)

  contains_invalid_char <- grepl("[^nwra]", text_before_slash)

  if (length(filters_input[contains_slash & contains_invalid_char]) > 0) {
    return(
      paste0(
        "At least one filter expression refers to an unknown object type: ",
        paste(
          paste0(
            "'", filters_input[contains_slash & contains_invalid_char], "'"
          ),
          collapse = ", "
        ),
        ". Valid object types are 'n', 'w', 'r' and 'a'"
      )
    )
  }

  return(TRUE)
}

assert_filters <- checkmate::makeAssertionFunction(check_filters)
