\name{Example3}
\alias{Example3}
\title{
Subgroup search in a clinical trial with a time-to-event endpoint 
}
\description{
Consider a clinical trial that was conducted to evaluate the efficacy and safety of an experimental treatment compared to a control treatment with a time-to-event primary endpoint (see the \link{survival} data set). A larger value of this endpoint indicates beneficial effect, e.g., longer survival. As in the other two examples, the trial's data set includes five biomarkers (four numeric biomarkers and one nominal biomarker) in addition to several covariates that can be included in the primary analysis model. The trial's sponsor is interested in identifying a set of promising subgroups with enhanced treatment effect. Subgroup search will be performed using the basic SIDES procedure and two SIDEScreen procedures (Fixed and Adaptive SIDEScreen procedures).\cr
The endpoint parameters will be defined as follows. The endpoint measures the time to an event of interest and thus the \code{type} parameter is set to \code{"survival"}. The names of the outcome and censoring variables need to be defined using \code{outcome_variable} and \code{outcome_censor_variable}, e.g., \code{outcome_variable = "outcome"} and \code{outcome_censor_variable = "outcome_censor"}. The value of the censoring variable that corresponds to a censored observation is defined using \code{outcome_censor_value}, e.g., \code{outcome_censor_value = "1"} if the value of 1 indicates that the patient did not experience the event of interest by the end of the treatment period. Since a larger value of the primary endpoint is associated with beneficial effect, the \code{direction} parameter is set to 1. \cr
Two sets of endpoint parameters will be defined to implement a simple assessment of the treatment effect based on the log-rank test as well as an advanced analysis based on a Cox proportional hazards regression model with an adjustment for important prognostic covariates:
\itemize{
\item Analysis strategy 1: The \code{analysis_method} parameter is set to \code{"Log-rank test"}.
\item Analysis strategy 2: The \code{analysis_method} parameter is set to \code{"Cox regression"}. The covariates to be included in this model need to be defined using the \code{cont_covariates} and \code{class_covariates} arguments. To adjust for two continuous covariates (cont1, cont2) and two class/categorical covariates (class1, class2), \code{cont_covariates} is set to \code{"cont1, cont2"} and \code{class_covariates} is set to \code{"class1, class2"}.
}
The data set parameters will be specified as follows:
\itemize{
\item The data set's name (\code{data_set}) is \code{survival}.
\item The treatment variable's name (\code{treatment_variable_name}) is \code{"treatment"}.
\item The value of the treatment variable that defines the control arm \cr (\code{treatment_variable_control_value}) is \code{"0"}.
\item The list of candidate biomarkers to be included in the subgroup search (\code{biomarker_names}) is \code{c("biomarker1", "biomarker2", "biomarker3", "biomarker4", "biomarker5")}.
\item The list of biomarker types (\code{biomarker_types}) is \code{c(rep("numeric", 4), "nominal")}.
}
The following algorithm parameters will be used in the subgroup search procedures:
\itemize{
\item Search depth (\code{depth}) is 2, which means that patient subgroups will be defined in terms of one or two biomarkers. Note that this is the default value of this parameter and thus it does not need to be explicitly defined.
\item Search width (\code{width}) is 2, i.e., only up to two best child subgroups will be retained for each parent group. This is the default value of this parameter and thus it does not need to be explicitly defined.
\item Child-to-parent ratio (\code{gamma}) equals 1 for each search level. This is the default value of this parameter and thus it does not need to be explicitly defined.
\item Minimum total number of patients in a promising subgroup (\code{min_subgroup_size}) is 60.
\item Minimum number of unique values for continuous biomarkers for applying a percentile transformation (\code{nperc}) is 20, i.e., a percentile transformation will not be applied to a continuous biomarker if there are few than 20 unique values. This is the default value of this parameter and thus it does not need to be explicitly defined.
\item Number of permutations to compute multiplicity-adjusted treatment effect p-values within promising subgroups (\code{n_perms_mult_adjust}) is 10. 
\item Number of processor cores to be used in parallel calculations (\code{ncores}) is 1. 
}
The subgroup search will be performed using the basic SIDES procedure as well as two SIDEScreen procedures by calling the \link{SubgroupSearch} function and a comprehensive summary of subgroup search results will be generated by calling the \link{GenerateReport} function.
}

\seealso{
  \code{\link{Example1}, \link{Example2}}
}
\examples{
  \donttest{
    ##############################################################################

    # Primary endpoint parameters

    # Analysis strategy 1: Analysis of the continuous endpoint without 
    # accounting for any covariates
    endpoint_parameters = list(outcome_variable = "outcome", 
      outcome_censor_variable = "outcome_censor",
      outcome_censor_value = "1",
      type = "survival",
      label = "Outcome", 
      analysis_method = "Log-rank test", 
      direction = 1)

    # Analysis strategy 2: Analysis of the continuous endpoint using a Cox model 
    # that accounts for two continuous covariates (cont1, cont2) and 
    # two class/categorical covariates (class1, class2)
    endpoint_parameters = list(outcome_variable = "outcome", 
      outcome_censor_variable = "outcome_censor",
      outcome_censor_value = "1",
      type = "survival",
      label = "Outcome", 
      analysis_method = "Cox regression", 
      cont_covariates = "cont1, cont2", 
      class_covariates = "class1, class2", 
      direction = 1)

    ##############################################################################

    # Data set parameters

    # Set of candidate biomarkers
    biomarker_names = c("biomarker1", "biomarker2", 
                        "biomarker3", "biomarker4", 
                        "biomarker5")

    # Biomarker type 
    biomarker_types = c(rep("numeric", 4), "nominal")

    # Data set parameters
    data_set_parameters = list(data_set = survival,
      treatment_variable_name = "treatment",
      treatment_variable_control_value = "0",
      biomarker_names = biomarker_names,
      biomarker_types = biomarker_types)

    ##############################################################################

    # Algorithm parameters for the basic SIDES procedure

    # Algorithm
    subgroup_search_algorithm = "SIDES procedure"

    # Number of permutations to compute multiplicity-adjusted treatment 
    # effect p-values within promising subgroups
    n_perms_mult_adjust = 10

    # Number of processor cores (use less or equal number of CPU cores on the current host)
    ncores = 1

    # Default values for the search depth (2), search width (2), 
    # maximum number of unique values for continuous biomarkers (20)

    # Algorithm parameters
    algorithm_parameters = list(
      n_perms_mult_adjust = n_perms_mult_adjust,
      min_subgroup_size = 60,
      subgroup_search_algorithm = subgroup_search_algorithm,
      ncores = ncores,
      random_seed = 3011)

    # Perform subgroup search

    # List of all parameters
    parameters = list(endpoint_parameters = endpoint_parameters,
      data_set_parameters = data_set_parameters,
      algorithm_parameters = algorithm_parameters)

    results = SubgroupSearch(parameters)

    # Simple summary of subgroup search results
    results

    # Generate a detailed Word-based report with a summary of subgroup search results
    report_information = GenerateReport(results,
      report_title = "Subgroup search report", 
      report_filename = tempfile(
        "Time-to-event endpoint (SIDES).docx", 
        fileext=".docx"
      )
    )

    ##############################################################################

    # Algorithm parameters for the Fixed SIDEScreen procedure

    # Algorithm
    subgroup_search_algorithm = "Fixed SIDEScreen procedure"

    # Number of permutations to compute multiplicity-adjusted treatment 
    # effect p-values within promising subgroups
    n_perms_mult_adjust = 10

    # Number of processor cores (use less or equal number of CPU cores on the current host)
    ncores = 1

    # Number of biomarkers selected for the second stage in the Fixed SIDEScreen algorithm
    n_top_biomarkers = 3

    # Default values for the search depth (2), search width (2), 
    # maximum number of unique values for continuous biomarkers (20)

    # Algorithm parameters
    algorithm_parameters = list(
      n_perms_mult_adjust = n_perms_mult_adjust,
      min_subgroup_size = 60,
      subgroup_search_algorithm = subgroup_search_algorithm,
      ncores = ncores,
      n_top_biomarkers = n_top_biomarkers,
      random_seed = 3011)

    # Perform subgroup search

    # List of all parameters
    parameters = list(endpoint_parameters = endpoint_parameters,
      data_set_parameters = data_set_parameters,
      algorithm_parameters = algorithm_parameters)

    results = SubgroupSearch(parameters)

    # Simple summary of subgroup search results
    results

    # Generate a detailed Word-based report with a summary of subgroup search results
    report_information = GenerateReport(results,
      report_title = "Subgroup search report", 
      report_filename = tempfile(
        "Time-to-event endpoint (Fixed SIDEScreen).docx", 
        fileext=".docx"
      )
    )

    ##############################################################################

    # Algorithm parameters for the Adaptive SIDEScreen procedure

    # Algorithm
    subgroup_search_algorithm = "Adaptive SIDEScreen procedure"

    # Number of permutations to compute multiplicity-adjusted treatment 
    # effect p-values within promising subgroups
    n_perms_mult_adjust = 10

    # Number of processor cores (use less or equal number of CPU cores on the current host)
    ncores = 1

    # Multiplier for selecting biomarkers for the second stage 
    # in the Adaptive SIDEScreen algorithm
    multiplier = 1

    # Number of permutations for computing the null distribution 
    # of the maximum VI score in the Adaptive SIDEScreen algorithm
    n_perms_vi_score = 100

    # Default values for the search depth (2), search width (2), 
    # maximum number of unique values for continuous biomarkers (20)

    # Algorithm parameters
    algorithm_parameters = list(
      n_perms_mult_adjust = n_perms_mult_adjust,
      min_subgroup_size = 60,
      subgroup_search_algorithm = subgroup_search_algorithm,
      ncores = ncores,
      multiplier = multiplier,
      n_perms_vi_score = n_perms_vi_score,
      random_seed = 3011)

    # Perform subgroup search

    # List of all parameters
    parameters = list(endpoint_parameters = endpoint_parameters,
      data_set_parameters = data_set_parameters,
      algorithm_parameters = algorithm_parameters)

    results = SubgroupSearch(parameters)

    # Simple summary of subgroup search results
    results

    # Generate a detailed Word-based report with a summary of subgroup search results
    GenerateReport(results,
      report_title = "Subgroup search report", 
      report_filename = tempfile(
        "Time-to-event endpoint (Adaptive SIDEScreen).docx", 
        fileext=".docx"
      )
    )
  }
}
