% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/call_rtmpt.R
\name{fit_drtmpt}
\alias{fit_drtmpt}
\title{Fit Diffusion-RT-MPT Models}
\usage{
fit_drtmpt(
  model,
  data,
  n.chains = 4,
  n.iter = 1000,
  n.phase1 = 1000,
  n.phase2 = 2000,
  n.thin = 1,
  Rhat_max = 1.1,
  Irep = 1000,
  prior_params = NULL,
  flags = NULL,
  control = NULL
)
}
\arguments{
\item{model}{A list of the class \code{drtmpt_model}.}

\item{data}{Optimally, a list of class \code{drtmpt_data}. Also possible is a \code{data.frame} or a
path to the text file. Both, \code{data.frame} and the text file must contain the column names "subj",
"group", "tree", "cat", and "rt" preferably but not necessarily in this order. The values of the latter must
be in milliseconds. It is always advised to use \code{\link{to_drtmpt_data}} first, which gives back a
\code{drtmpt_data} list with information about the changes in the data, that were needed.}

\item{n.chains}{Number of chains to use. Default is 4. Must be larger than 1 and smaller or equal to 16.}

\item{n.iter}{Number of samples per chain. Default is 1000.}

\item{n.phase1}{Number of samples for phase 1 (adaptation phase). Default is 1000.}

\item{n.phase2}{Number of samples for phase 2. Default is 2000.}

\item{n.thin}{Thinning factor. Default is 1.}

\item{Rhat_max}{Maximal Potential scale reduction factor: A lower threshold that needs to be reached before the actual sampling starts. Default is 1.05}

\item{Irep}{Every \code{Irep} samples an interim state with the current maximal potential scale reduction
factor is shown. Default is 1000. The following statements must hold true for \code{Irep}:
\itemize{
  \item \code{n.phase1} is a multiple of \code{Irep}
  \item \code{n.phase2} is a multiple of \code{Irep}
  \item \code{n.phase1} is smaller than or equal to \code{n.phase2},
  \item \code{Irep} is a multiple of \code{n.thin} and
  \item \code{n.iter} is a multiple of \code{Irep / n.thin}.
}}

\item{prior_params}{Named list with prior parameters. All parameters have default values, that lead to uninformative priors. Vectors are not allowed.
Allowed parameters are:
\itemize{
  \item \code{prec_epsilon}: prior precision for population means of process-related parameters.
    Default is \code{1.0}.
  \item \code{delta_df}: degrees of freedom of t-distribution for motor times. Default is \code{10}.
  \item \code{delta_mu}: mean of t-distribution for motor times. Default is \code{0.5}.
  \item \code{delta_scale}: scale of t-distribution for motor times. Default is \code{1.0}.
  \item \code{SIGMA_Corr_eta}: shape parameter for LKJ distribution for process-related parameters. Default is \code{4.0}.
  \item \code{SIGMA_SD_rho}: scale parameter of half-Cauchy distribution for process-related parameters. Default is \code{2.5}.
  \item \code{GAMMA_Corr_eta}: shape parameter for LKJ distribution for motor-related parameters. Default is \code{4.0}.
  \item \code{GAMMA_SD_rho}: scale parameter of half-Cauchy distribution for motor-related parameters. Default is \code{0.5}.
  \item \code{Omega2_alpha}: shape parameter of gamma distribution for residual variance. Default is \code{0.0025}.
  \item \code{Omega2_beta}: rate parameter of gamma distribution for residual variance. Default is \code{0.5}.
}}

\item{flags}{Either NULL or a list of
\itemize{
  \item \code{old_label} If set to \code{TRUE} the old labels of "subj" and "group" of the data will be used in the elements of the output list.
    Default is \code{FALSE}.
  \item \code{indices} Model selection indices. If set to \code{TRUE} the log-likelihood for each iteration and trial will be
    stored temporarily and with that the DIC, as well as the WAIC and LOOIC will be calculated via the \code{loo} package. If you want to have the
    log-likelihood matrix stored in the output of this function, you can set \code{loglik} to \code{TRUE}. Default for
    \code{indices} is \code{FALSE}.
  \item \code{loglik} If set to \code{TRUE} and \code{indices = TRUE} the log-likelihood matrix for each iteration and trial will
    be saved in the output as a matrix. Default is \code{FALSE}.
  \item \code{random_init} If set to \code{TRUE} the initial values are randomly drawn. If \code{FLASE} maximum likelihood is used
    for initial values.
}}

\item{control}{Either NULL or a list of
\itemize{
  \item \code{maxthreads} for the ML estimation of the initial values and the calculation of the DIC values one can use more than
    \code{n.chains} threads for parallelization. Default is 4 like \code{n.chians}. \code{maxthreads} must be larger or equal to
    \code{n.chains}.
  \item \code{maxtreedepth1_3} maxtree-depth of the no-U-turn algorithm in Phases 1 to 3
  \item \code{maxtreedepth4} maxtree-depth of the no-U-turn algorithm in Phases 4
}}
}
\value{
A list of the class \code{drtmpt_fit} containing
  \itemize{
    \item \code{samples}: the posterior samples as an \code{mcmc.list} object,
    \item \code{diags}: some diagnostics like deviance information criterion, posterior predictive checks for the frequencies and latencies,
                        potential scale reduction factors, and also the 99\% and 95\% HDIs and medians for the group-level parameters,
    \item \code{specs}: some model specifications like the model, arguments of the model call, and information about the data transformation,
    \item \code{indices} (optional): if enabled, WAIC and LOO,
    \item \code{LogLik} (optional): if enabled, the log-likelihood matrix used for WAIC and LOO.
    \item \code{summary} includes posterior mean and median of the main parameters.
  }
}
\description{
Given model and data, this function a Hamiltonian MCMC sampler and stores the samples in an mcmc.list called \code{samples}.
  Posterior predictive checks developed by Klauer (2010), deviance information criterion (DIC; Spiegelhalter et al., 2002),
  99\% and 95\% highest density intervals (HDI) together with the median will be provided for the main parameters in a list
  called \code{diags}. Optionally, the \code{indices} widely applicable information criterion (WAIC; Watanabe, 2010; Vehtari et al., 2017) and
  leave-one-out cross-validation (LOO; Vehtari et al., 2017) can be saved. Additionally the log-likelihood (\code{LogLik}) can also be stored.
  Some specifications of the function call are also saved in \code{specs}.
}
\examples{
####################################################################################
# Detect-Guess variant of the Two-High Threshold model.
# The encoding and motor execution times are assumed to be equal for each response.
####################################################################################

mdl_2HTM <- "
# targets
do+(1-do)*g
(1-do)*(1-g)

# lures
(1-dn)*g
dn+(1-dn)*(1-g)

# do: detect old; dn: detect new; g: guess
"

model <- to_drtmpt_model(mdl_file = mdl_2HTM)

data_file <- system.file("extdata/data.txt", package="rtmpt")
data <- read.table(file = data_file, header = TRUE)
data_list <- to_drtmpt_data(raw_data = data, model = model)
\donttest{
# This might take some time
drtmpt_out <- fit_drtmpt(model = model, data = data_list, Rhat_max = 1.1)
drtmpt_out
}
}
\references{
Klauer, K. C. (2010). Hierarchical multinomial processing tree models: A latent-trait approach. \emph{Psychometrika, 75(1)}, 70-98.

Spiegelhalter, D. J., Best, N. G., Carlin, B. P., & Van Der Linde, A. (2002). Bayesian measures of model complexity and fit.
  \emph{Journal of the royal statistical society: Series b (statistical methodology), 64(4)}, 583-639.

Vehtari, A., Gelman, A., & Gabry, J. (2017). Practical Bayesian model evaluation using leave-one-out cross-validation and WAIC.
  \emph{Statistics and Computing, 27(5)}, 1413-1432.

Watanabe, S. (2010). Asymptotic equivalence of Bayes cross validation and widely applicable information criterion in singular learning theory.
  \emph{Journal of Machine Learning Research, 11(Dec)}, 3571-3594.
}
\author{
Raphael Hartmann
}
