% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/generate-clusters.R
\name{genclust}
\alias{genclust}
\title{Generate clusters for spatial clustering}
\usage{
genclust(x, nclust = 10, weights = NULL)
}
\arguments{
\item{x}{An \code{sf} or \code{sfc} object representing spatial polygonal data. It can also be
a \code{matrix} or \code{Matrix} object with non-zero values representing weighted
connectivity between units.}

\item{nclust}{Integer, specifying the initial number of clusters.}

\item{weights}{Optional \code{numeric} vector or \code{matrix} of weights between units in \code{x}.
It should have dimensions \code{n^2}, where \code{n} is the number of units in \code{x}. If NULL,
random weights are assigned.}
}
\value{
A list with three elements:
\itemize{
\item \code{graph}: The undirected graph object representing spatial contiguity.
\item \code{mst}: The minimum spanning tree.
\item \code{membership}: The cluster membership for elements in \code{x}.
}
}
\description{
Creates an undirected graph from spatial polygonal data, computes its minimum spanning
tree (MST), and generates \code{nclust} clusters. This function is used to initialize
cluster membership in a clustering algorithm, such as \code{sfclust}.
}
\examples{

library(sfclust)
library(sf)

x <- st_make_grid(cellsize = c(1, 1), offset = c(0, 0), n = c(3, 2))

# using distance between geometries
clust <- genclust(x, nclust = 3, weights = st_distance(st_centroid(x)))
print(clust)
plot(st_sf(x, cluster = factor(clust$membership)))

# using increasing weights
cluster_ini <- genclust(x, nclust = 3, weights = 1:36)
print(cluster_ini)
plot(st_sf(x, cluster = factor(cluster_ini$membership)))

# using on random weights
cluster_ini <- genclust(x, nclust = 3, weights = runif(36))
print(cluster_ini)
plot(st_sf(x, cluster = factor(cluster_ini$membership)))

}
