% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eigengap.R
\name{sgdgmf.rank}
\alias{sgdgmf.rank}
\title{Rank selection via eigenvalue-gap methods}
\usage{
sgdgmf.rank(
  Y,
  X = NULL,
  Z = NULL,
  maxcomp = ncol(Y),
  family = gaussian(),
  weights = NULL,
  offset = NULL,
  method = c("evr", "onatski", "act", "oht"),
  type.reg = c("ols", "glm"),
  type.res = c("deviance", "pearson", "working", "link"),
  normalize = FALSE,
  maxiter = 10,
  parallel = FALSE,
  nthreads = 1,
  return.eta = FALSE,
  return.mu = FALSE,
  return.res = FALSE,
  return.cov = FALSE
)
}
\arguments{
\item{Y}{matrix of responses (\eqn{n \times m})}

\item{X}{matrix of row-specific fixed effects (\eqn{n \times p})}

\item{Z}{matrix of column-specific fixed effects (\eqn{q \times m})}

\item{maxcomp}{maximum number of eigenvalues to compute}

\item{family}{a family as in the \code{\link{glm}} interface (default \code{gaussian()})}

\item{weights}{matrix of optional weights (\eqn{n \times m})}

\item{offset}{matrix of optional offsets (\eqn{n \times m})}

\item{method}{rank selection method}

\item{type.reg}{regression method to be used to profile out the covariate effects}

\item{type.res}{residual type to be decomposed}

\item{normalize}{if \code{TRUE}, standardize column-by-column the residual matrix}

\item{maxiter}{maximum number of iterations}

\item{parallel}{if \code{TRUE}, allows for parallel computing using \code{foreach}}

\item{nthreads}{number of cores to be used in parallel (only if \code{parallel=TRUE})}

\item{return.eta}{if \code{TRUE}, return the linear predictor martix}

\item{return.mu}{if \code{TRUE}, return the fitted value martix}

\item{return.res}{if \code{TRUE}, return the residual matrix}

\item{return.cov}{if \code{TRUE}, return the covariance matrix of the residuals}
}
\value{
A list containing the \code{method}, the selected latent rank \code{ncomp},
and the eigenvalues used to select the latent rank \code{lambdas}.
Additionally, if required, in the output list will also provide the linear predictor
\code{eta}, the predicted mean matrix \code{mu}, the residual matrix \code{res}, and
the implied residual covariance matrix \code{covmat}.
}
\description{
Select the number of significant principal components of a GMF model via
exploitation of eigenvalue-gap methods
}
\examples{
library(sgdGMF)

# Set the data dimensions
n = 100; m = 20; d = 5

# Generate data using Poisson, Binomial and Gamma models
data_pois = sim.gmf.data(n = n, m = m, ncomp = d, family = poisson())
data_bin = sim.gmf.data(n = n, m = m, ncomp = d, family = binomial())
data_gam = sim.gmf.data(n = n, m = m, ncomp = d, family = Gamma(link = "log"), dispersion = 0.25)

# Initialize the GMF parameters assuming 3 latent factors
ncomp_pois = sgdgmf.rank(data_pois$Y, family = poisson(), normalize = TRUE)
ncomp_bin = sgdgmf.rank(data_bin$Y, family = binomial(), normalize = TRUE)
ncomp_gam = sgdgmf.rank(data_gam$Y, family = Gamma(link = "log"), normalize = TRUE)

# Get the selected number of components
print(paste("Poisson:", ncomp_pois$ncomp))
print(paste("Binomial:", ncomp_bin$ncomp))
print(paste("Gamma:", ncomp_gam$ncomp))

# Plot the screeplot used for the component determination
oldpar = par(no.readonly = TRUE)
par(mfrow = c(3,1))
barplot(ncomp_pois$lambdas, main = "Poisson screeplot")
barplot(ncomp_bin$lambdas, main = "Binomial screeplot")
barplot(ncomp_gam$lambdas, main = "Gamma screeplot")
par(oldpar)

}
\references{
Onatski, A. (2010).
\emph{Determining the number of factors from empirical distribution of eigenvalues.}
Review of Economics and Statistics, 92(4): 1004-1016

Ahn, S.C., Horenstein, A.R. (2013).
\emph{Eigenvalue ratio test for the number of factors.}
Econometrica, 81, 1203-1227

Gavish, M., Donoho, D.L. (2014)
\emph{The optimal hard thresholding for singular values is 4/sqrt(3).}
IEEE Transactions on Information Theory, 60(8): 5040--5053

Fan, J., Guo, J. and Zheng, S. (2020).
\emph{Estimating number of factors by adjusted eigenvalues thresholding.}
Journal of the American Statistical Association, 117(538): 852--861

Wang, L. and Carvalho, L. (2023).
\emph{Deviance matrix factorization.}
Electronic Journal of Statistics, 17(2): 3762-3810
}
