#' Tidy the result of a time-series model into a summary tibble
#'
#'
#' @param x An object to be converted into a tibble ("tidy" data.frame)
#' @param ... extra arguments
#'
#' @details
#' `sw_tidy()` is a wrapper for [broom::tidy()]. The main benefit of `sw_tidy()`
#' is that it has methods for various time-series model classes such as
#' `HoltWinters`, `ets`, `Arima`, etc.
#' `sw_tidy()` methods always returns a "tidy" tibble with model coefficient / parameters.
#'
#' For non-time series, `sw_tidy()` defaults to [broom::tidy()].
#' The only difference is that the return is a tibble.
#' The output of `sw_tidy()` is always a tibble with disposable row names. It is
#' therefore suited for further manipulation by packages like dplyr and
#' ggplot2.
#'
#' @seealso [broom::tidy()]
#'
#' @return a tibble
#'
#' @examples
#' library(dplyr)
#' library(forecast)
#'
#' WWWusage %>%
#'     auto.arima() %>%
#'     sw_tidy(conf.int = TRUE)
#'
#' @export
sw_tidy <- function(x, ...) UseMethod("sw_tidy")


#' Default tidying method
#'
#' By default, `sw_tidy()` uses [broom::tidy()] to convert its output.
#'
#'
#' @param x an object to be tidied
#' @param ... extra arguments passed to [broom::tidy()]
#'
#' @return A tibble generated by [broom::tidy()]
#'
#' @export
sw_tidy.default <- function(x, ...) {
    broom::tidy(x, ...) %>%
        tibble::as_tibble()
}
