% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tdROC.cr.R
\name{tdROC.cr}
\alias{tdROC.cr}
\title{Estimate time-dependent prediction accuracy measures, including the ROC, AUC, Brier score, and survival probability difference, with competing risk data.}
\usage{
tdROC.cr(
  X,
  Y,
  delta,
  tau,
  span = 0.1,
  h = NULL,
  type = "uniform",
  epsilon = 0.01,
  cut.off = NULL,
  n.grid = 1000,
  nboot = 0,
  alpha = 0.05,
  method = "both",
  output = "both"
)
}
\arguments{
\item{X}{a numeric vector of risk score in the same length as \code{Y} and \code{delta}, such as biomarker or predicted probability. A higher value indicates higher risk of the event.
The calibration results (Brier score, survival difference) are applicable only when the risk score has the predicted probability interpretation.}

\item{Y}{a numeric vector of time to event in the same length as \code{X} and \code{delta}.}

\item{delta}{a vector of numeric indicator of event type in the same length as \code{X} and \code{Y}. The primary event should be coded as 1,
the competing event should be coded as 2, and censoring should be coded as 0.}

\item{tau}{a scalar, the prediction horizon at which the prediction is evaluated.}

\item{span}{a numeric value, the proportion of neighbour observations used in nearest neighbor method. The default is 0.1.}

\item{h}{a numeric value, the bandwidth of kernel weights, the defualt is \code{NULL}. If not specified, the function will use the value of
\code{span} to calculate kernel weights. In case both \code{span} and \code{h} are specified, the function will use \code{h}.}

\item{type}{a character value, indicating the type of kernel function used to calculate kernel weights. The default is \code{"uniform"} kernel. Other options are \code{"Epanechnikov"} and \code{"normal"}.
It will only be used when the bandwidth \code{h} is specified.}

\item{epsilon}{The precision parameter used in an approximation to the weight calculation when the sample size is large. If a weight corresponding to a specific risk score is already calculated,
then the weights corresponding to adjacent risk scores, within the distance specified by epsilon, will be the same under the approximation. This approximation avoids repeated
calculation of weights that are almost the same, and hence increases the speed of computation in this situation. The default is NULL, which means no approximation is used. A higher
value indicates less precision.}

\item{cut.off}{a vector of \code{X} cut-off values at which sensitivity and specificity will be calculated.}

\item{n.grid}{a positive integer, the number of grid points used when calculating the ROC curve. The default is \code{1000}.}

\item{nboot}{the number of bootstrap replications to be used for variance estimation. The default is \code{nboot = 0}, corresponding to no variance estimation.}

\item{alpha}{It is (1 - level of confidence interval)/2, default is \code{0.05}. It is used only when \code{nboot > 0}.}

\item{method}{It is used to specify which method you would like to use to estimate AUC, default to \code{"both"}. Other options are \code{"integral"} and \code{"empirical"}.}

\item{output}{It is used to specify which kind of output you want, default to \code{"both"}. Other options are \code{"AUC"}, including AUC, sensitivity, and specificity are included,
and \code{"calibration"} including Brier Score and survival difference.}
}
\value{
Returns a list of the following items:

\code{main_res:} a list of \code{AUC.A.integral} estimated by trapezoidal integration for definition A,
       \code{AUC.A.empirical} estimated by empirical estimator for definition A,
        \code{AUC.B.integral} estimated by trapezoidal integration for definition B,
       \code{AUC.B.empirical} estimated by empirical estimator for definition B,
       and a data frame \code{ROC} with dimension \code{(2+n.grid) x 4} with columns \code{cut.off}, \code{sens}, \code{specA} and \code{specB}.

\code{calibration_res:} brier score and survival difference estimated based on the formula similar to Wu and Li (2018). When the risk score \code{X}
is a biomarker value instead of a predicted cumulative incidence probability, the brier score and survival difference cannot be calculated. In this case, please disregard the calibration results.

\code{boot_res:} a list of bootstrap results, including \code{bAUC.A.integral}, \code{bAUC.A.empirical},  \code{bAUC.B.integral}, \code{bAUC.B.empirical}, \code{bBS}, \code{bSurvDiff}, \code{bROC}.
 For \code{bAUC.A.integral}, \code{bAUC.A.empirical},  \code{bAUC.B.integral}, \code{bAUC.B.empirical}, \code{bBS}, \code{bSurvDiff}, each one is a list including corresponding mean, standard deviation, confidence interval.
 \code{bROC} is a data frame with colomns \code{sens.mean}, \code{sens.sd}, \code{sens.lower}, \code{sens.upper},
 \code{specA.mean}, \code{specA.sd}, \code{specA.lower}, \code{specA.upper}, \code{specB.mean}, \code{specB.sd}, \code{specB.lower}, \code{specB.upper}
}
\description{
This is a core function of the ‘tdROC‘ package. It uses the nonparametric weights proposed by Wu (Wu and Li, 2018)
to estimate a number of time-dependent prediction accuracy measures for right-censored survival outcomes,
including ROC curve, AUC, Brier score, and survival difference, with competing risk data. For each measure, the variance can be estimated
through bootstrap resampling.
}
\details{
This function takes the risk score value \code{X}, the time-to-event data \code{Y} and censoring indicator \code{delta} as input to estimate
 a number of time-dependent prediction accuracy measures for survival outcomes, including ROC curve, AUC, Brier score, and survival difference, with competing risk.
 The confidence intervals of above quantities are estimated by bootstrap.

 For competing risk data, there are two definition of controls introduced by Zheng et al, which are listed below

 \deqn{
 \text{Definition A:} \text{Case} k:T \le \tau, \delta = k; \text{Control}_A: (T>\tau)\cup (T \le \tau \cap \delta \ne k)
 }

 \deqn{
 \text{Definition B:} \text{Case} k:T \le \tau, \delta = k; \text{Control}_B: (T>\tau)
 }

 Based on the definition A, both the event-free subjects and subjects who experience other competing events were included as controls. While definition B include only event-free subjects.
 This function offers two options to estimate AUC. The first one make use of estimated sensitivity and specificity to calculate the AUC via trapezoidal integration
 by setting a series of cutoff point. For the two different definition, we separately calculate the sensitivity, specificity and AUC. The output will also include the sensitivities
 and specificities for our plot function. The other one estimates AUC by the empirical estimator of the proportion of concordance pairs with proposed weight estimator (Wu and Li, 2018).
 These two methods generate quite similar estimates. The option can be set by the argument \code{method}.

 In addition to the above prediction measures, we include Brier Score and survival difference to evaluate the calibration metrics. Their definitions are included below.
 They can be estimated with the proposed conditional probability weight (Wu and Li, 2018).
 Both of them are measures to assess the accuracy of probabilistic predictions \code{X}. The calibration result makes sense only
 when the risk score \code{X} is a predicted probability, and should be ignored otherwise.

 \deqn{
 \text{Brier Score} = E{[1(T \le \tau, \delta = 1) - X]^2}
 }

 \deqn{
 \text{Survival difference} = E[1(T \le \tau, \delta = 1) - X]
 }

 This function uses the same approximation as \code{tdROC} with the argument \code{epsilon}
}
\examples{
library(survival)
data(Melano)
expit <- function(x){ 1/(1+exp(-x)) }
tdROC.cr(X = expit(Melano$thick) , Y = Melano$time, delta = Melano$status, tau = 1800, nboot = 10)

}
\references{
Zheng Y, Cai T, Jin Y, Feng Z. Evaluating prognostic accuracy of biomarkers under competing risk. Biometrics. 2012;68(2):388-396. doi:10.1111/j.1541-0420.2011.01671.x
}
