
<!-- README.md is generated from README.Rmd. Please edit that file -->

# tfarima

<!-- badges: start -->

[![CRAN
status](https://www.r-pkg.org/badges/version/tfarima)](https://CRAN.R-project.org/package=tfarima)
[![Downloads](https://cranlogs.r-pkg.org/badges/grand-total/tfarima)](https://cran.r-project.org/package=tfarima)
[![Downloads per
month](https://cranlogs.r-pkg.org/badges/last-month/tfarima)](https://cran.r-project.org/package=tfarima)
<!-- badges: end -->

## Overview

**tfarima** provides a comprehensive framework for building customized
**Transfer Function and ARIMA models** with multiple operators and
parameter restrictions. The package implements exact maximum likelihood
estimation and offers a wide range of tools for time series analysis.

### Key Features

- **Flexible model specification**: Build ARIMA and transfer function
  models with multiple lag polynomials and parameter restrictions
- **Model identification**: Tools for identifying appropriate model
  structures
- **Estimation methods**: Exact or conditional maximum likelihood
  estimation
- **Diagnostic checking**: Comprehensive model validation tools
- **Automatic outlier detection**: Identify and handle outliers in time
  series
- **Calendar effects**: Model trading day, Easter, and leap year effects
- **Forecasting**: Generate predictions with confidence intervals
- **Seasonal adjustment**: Decompose time series into trend, seasonal,
  and irregular components

## Installation

Install the stable version from CRAN:

``` r
install.packages("tfarima")
```

Or install the development version from GitHub:

``` r
# install.packages("devtools")
devtools::install_github("gallegoj/tfarima")
```

## Quick Start

``` r
library(tfarima)

# Load example data
data(rsales)

# Build a basic ARIMA model with seasonal components
model <- um(rsales, 
            i = list(1, c(1, 12)),        # Regular and seasonal differences
            ma = list(1, c(1, 12)),       # MA(1) and seasonal MA(1)
            bc = TRUE)                     # Box-Cox transformation

# Fit the model
fitted_model <- fit(model)

# Display results
summary(fitted_model)

# Generate forecasts
predictions <- predict(fitted_model, n.ahead = 12)
plot(predictions)

# Diagnostic checking
tsdiag(fitted_model)
```

## Main Functions

### Model Building

- `um()`: Build univariate ARIMA models
- `tfm()`: Build transfer function models
- `lagpol()`: Create lag polynomials with parameter restrictions

### Model Estimation and Selection

- `fit()`: Estimate model parameters using exact or conditional ML
- `outliers()`: Detect and model outliers automatically
- `calendar()`: Add calendar effects (trading days, Easter, leap year)

### Model Evaluation

- `tsdiag()`: Diagnostic checking plots
- `residuals()`: Extract model residuals
- `AIC()`, `logLik()`: Information criteria and likelihood

### Forecasting and Decomposition

- `predict()`: Generate forecasts
- `seasadj()`: Seasonal adjustment
- `ucomp()`: Unobserved components decomposition

## Example: Seasonal Adjustment

``` r
# Load retail sales data
data(rsales)

# Build and fit model with calendar effects
model <- um(rsales, 
            i = list(1, c(1, 12)), 
            ma = list(1, c(1, 12)), 
            bc = TRUE)

# Add calendar effects
model_cal <- calendar(model, easter = TRUE)
fitted <- fit(model_cal)

# Perform seasonal adjustment
sa <- seasadj(fitted)

# Plot results
plot(sa)
```

## Example: Transfer Function Model

``` r
# Load gas furnace data
data(seriesJ)

# Build transfer function model
model <- tfm(seriesJ$output, 
             inputs = list(seriesJ$input),
             orders = list(c(3, 2, 0)))

# Fit the model
fitted <- fit(model)

# Summary and diagnostics
summary(fitted)
tsdiag(fitted)
```

## Documentation

For more detailed information and examples, see:

- Package vignette: `vignette("tfarima")`
- Function documentation: `help(package = "tfarima")`
- CRAN page: <https://CRAN.R-project.org/package=tfarima>

## References

The package implements methods from:

- Box, G.E.P., Jenkins, G.M., Reinsel, G.C. and Ljung, G.M. (2015).
  *Time Series Analysis: Forecasting and Control*. John Wiley & Sons.

- Box, G.E.P., Pierce, D.A. and Newbold, D.A. (1987). Estimating Trend
  and Growth Rates in Seasonal Time Series. *Journal of the American
  Statistical Association*, 82(397), 276-282.

- Bell, W.R. and Hillmer, S.C. (1983). Modeling Time Series with
  Calendar Variation. *Journal of the American Statistical Association*,
  78(383), 526-534.

- Chen, C. and Liu, L.M. (1993). Joint Estimation of Model Parameters
  and Outlier Effects in Time Series. *Journal of the American
  Statistical Association*, 88(421), 284-297.

## License

GPL (\>= 2)

## Author

José L. Gallego

## Issues and Contributions

To report bugs or request features, please visit:
<https://github.com/gallegoj/tfarima/issues>
