% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nn-conv.R
\name{nn_conv_transpose2d}
\alias{nn_conv_transpose2d}
\title{ConvTranpose2D module}
\usage{
nn_conv_transpose2d(
  in_channels,
  out_channels,
  kernel_size,
  stride = 1,
  padding = 0,
  output_padding = 0,
  groups = 1,
  bias = TRUE,
  dilation = 1,
  padding_mode = "zeros"
)
}
\arguments{
\item{in_channels}{(int): Number of channels in the input image}

\item{out_channels}{(int): Number of channels produced by the convolution}

\item{kernel_size}{(int or tuple): Size of the convolving kernel}

\item{stride}{(int or tuple, optional): Stride of the convolution. Default: 1}

\item{padding}{(int or tuple, optional): \code{dilation * (kernel_size - 1) - padding} zero-padding
will be added to both sides of each dimension in the input. Default: 0}

\item{output_padding}{(int or tuple, optional): Additional size added to one side
of each dimension in the output shape. Default: 0}

\item{groups}{(int, optional): Number of blocked connections from input channels to output channels. Default: 1}

\item{bias}{(bool, optional): If \code{True}, adds a learnable bias to the output. Default: \code{True}}

\item{dilation}{(int or tuple, optional): Spacing between kernel elements. Default: 1}

\item{padding_mode}{(string, optional): \code{'zeros'}, \code{'reflect'},
\code{'replicate'} or \code{'circular'}. Default: \code{'zeros'}}
}
\description{
Applies a 2D transposed convolution operator over an input image
composed of several input planes.
}
\details{
This module can be seen as the gradient of Conv2d with respect to its input.
It is also known as a fractionally-strided convolution or
a deconvolution (although it is not an actual deconvolution operation).
\itemize{
\item \code{stride} controls the stride for the cross-correlation.
\item \code{padding} controls the amount of implicit zero-paddings on both
sides for \code{dilation * (kernel_size - 1) - padding} number of points. See note
below for details.
\item \code{output_padding} controls the additional size added to one side
of the output shape. See note below for details.
\item \code{dilation} controls the spacing between the kernel points; also known as the à trous algorithm.
It is harder to describe, but this \code{link}_ has a nice visualization of what \code{dilation} does.
\item \code{groups} controls the connections between inputs and outputs.
\code{in_channels} and \code{out_channels} must both be divisible by
\code{groups}. For example,
\itemize{
\item At groups=1, all inputs are convolved to all outputs.
\item At groups=2, the operation becomes equivalent to having two conv
layers side by side, each seeing half the input channels,
and producing half the output channels, and both subsequently
concatenated.
\item At groups= \code{in_channels}, each input channel is convolved with
its own set of filters (of size
\eqn{\left\lfloor\frac{out\_channels}{in\_channels}\right\rfloor}).
}
}

The parameters \code{kernel_size}, \code{stride}, \code{padding}, \code{output_padding}
can either be:
\itemize{
\item a single \code{int} -- in which case the same value is used for the height and width dimensions
\item a \code{tuple} of two ints -- in which case, the first \code{int} is used for the height dimension,
and the second \code{int} for the width dimension
}
}
\note{
Depending of the size of your kernel, several (of the last)
columns of the input might be lost, because it is a valid \code{cross-correlation}_,
and not a full \code{cross-correlation}. It is up to the user to add proper padding.

The \code{padding} argument effectively adds \code{dilation * (kernel_size - 1) - padding}
amount of zero padding to both sizes of the input. This is set so that
when a \link{nn_conv2d} and a \link{nn_conv_transpose2d} are initialized with same
parameters, they are inverses of each other in
regard to the input and output shapes. However, when \code{stride > 1},
\link{nn_conv2d} maps multiple input shapes to the same output
shape. \code{output_padding} is provided to resolve this ambiguity by
effectively increasing the calculated output shape on one side. Note
that \code{output_padding} is only used to find output shape, but does
not actually add zero-padding to output.

In some circumstances when using the CUDA backend with CuDNN, this operator
may select a nondeterministic algorithm to increase performance. If this is
undesirable, you can try to make the operation deterministic (potentially at
a performance cost) by setting \code{torch.backends.cudnn.deterministic = TRUE}.
}
\section{Shape}{

\itemize{
\item Input: \eqn{(N, C_{in}, H_{in}, W_{in})}
\item Output: \eqn{(N, C_{out}, H_{out}, W_{out})} where
\deqn{
  H_{out} = (H_{in} - 1) \times \mbox{stride}[0] - 2 \times \mbox{padding}[0] + \mbox{dilation}[0]
\times (\mbox{kernel\_size}[0] - 1) + \mbox{output\_padding}[0] + 1
}
\deqn{
  W_{out} = (W_{in} - 1) \times \mbox{stride}[1] - 2 \times \mbox{padding}[1] + \mbox{dilation}[1]
\times (\mbox{kernel\_size}[1] - 1) + \mbox{output\_padding}[1] + 1
}
}
}

\section{Attributes}{

\itemize{
\item weight (Tensor): the learnable weights of the module of shape
\eqn{(\mbox{in\_channels}, \frac{\mbox{out\_channels}}{\mbox{groups}},}
\eqn{\mbox{kernel\_size[0]}, \mbox{kernel\_size[1]})}.
The values of these weights are sampled from
\eqn{\mathcal{U}(-\sqrt{k}, \sqrt{k})} where
\eqn{k = \frac{groups}{C_{\mbox{out}} * \prod_{i=0}^{1}\mbox{kernel\_size}[i]}}
\item bias (Tensor):   the learnable bias of the module of shape (out_channels)
If \code{bias} is \code{True}, then the values of these weights are
sampled from \eqn{\mathcal{U}(-\sqrt{k}, \sqrt{k})} where
\eqn{k = \frac{groups}{C_{\mbox{out}} * \prod_{i=0}^{1}\mbox{kernel\_size}[i]}}
}
}

\examples{
if (torch_is_installed()) {
# With square kernels and equal stride
m <- nn_conv_transpose2d(16, 33, 3, stride = 2)
# non-square kernels and unequal stride and with padding
m <- nn_conv_transpose2d(16, 33, c(3, 5), stride = c(2, 1), padding = c(4, 2))
input <- torch_randn(20, 16, 50, 100)
output <- m(input)
# exact output size can be also specified as an argument
input <- torch_randn(1, 16, 12, 12)
downsample <- nn_conv2d(16, 16, 3, stride = 2, padding = 1)
upsample <- nn_conv_transpose2d(16, 16, 3, stride = 2, padding = 1)
h <- downsample(input)
h$size()
output <- upsample(h, output_size = input$size())
output$size()
}
}
