% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/translator-io.R
\name{translator_read}
\alias{translator_read}
\alias{translator_write}
\alias{translations_read}
\alias{translations_write}
\alias{translations_paths}
\title{Read and Write Translations}
\usage{
translator_read(
  path = getOption("transltr.path"),
  encoding = "UTF-8",
  verbose = getOption("transltr.verbose", TRUE),
  translations = TRUE
)

translator_write(
  tr = translator(),
  path = getOption("transltr.path"),
  overwrite = FALSE,
  verbose = getOption("transltr.verbose", TRUE),
  translations = TRUE
)

translations_read(path = "", encoding = "UTF-8", tr = NULL)

translations_write(tr = translator(), path = "", lang = "")

translations_paths(
  tr = translator(),
  parent_dir = dirname(getOption("transltr.path"))
)
}
\arguments{
\item{path}{A non-empty and non-NA character string. A path to a file to
read from, or write to.
\itemize{
\item This file must be a Translator file for \code{\link[=translator_read]{translator_read()}}.
\item This file must be a translations file for \code{\link[=translations_read]{translations_read()}}.
}

See Details for more information. \code{\link[=translator_write]{translator_write()}} automatically
creates the parent directories of \code{path} (recursively) if they do not
exist.}

\item{encoding}{A non-empty and non-NA character string. The source character
encoding. In almost all cases, this should be UTF-8. Other encodings are
internally re-encoded to UTF-8 for portability.}

\item{verbose}{A non-NA logical value. Should progress information be
reported?}

\item{translations}{A non-NA logical value. Should translations files also
be read, or written along with \code{path} (the Translator file)?}

\item{tr}{A \code{\link{Translator}} object.

This argument is \code{NULL} by default for \code{\link[=translations_read]{translations_read()}}. If a
\code{\link{Translator}} object is passed to this function, it
will read translations and further register them (as long as they
correspond to an existing source text).}

\item{overwrite}{A non-NA logical value. Should existing files be
overwritten? If such files are detected and \code{overwrite} is set equal
to \code{TRUE}, an error is thrown.}

\item{lang}{A non-empty and non-NA character string. The underlying language.

A language is usually a code (of two or three letters) for a native
language name. While users retain full control over codes, it is best
to use language codes stemming from well-known schemes such as
\href{https://en.wikipedia.org/wiki/IETF_language_tag}{IETF BCP 47}, or
\href{https://en.wikipedia.org/wiki/List_of_ISO_639_language_codes}{ISO 639-1}
to maximize portability and cross-compatibility.}

\item{parent_dir}{A non-empty and non-NA character string. A path to a
parent directory.}
}
\value{
\code{\link[=translator_read]{translator_read()}} returns an \code{\link[R6:R6Class]{R6}} object of class
\code{\link{Translator}}.

\code{\link[=translator_write]{translator_write()}} returns \code{NULL}, invisibly. It is used for its
side-effects of
\itemize{
\item creating a Translator file to the location given by \code{path}, and
\item creating further translations file(s) in the same directory if
\code{translations} is \code{TRUE}.
}

\code{\link[=translations_read]{translations_read()}} returns an S3 object of class
\code{\link[=export]{ExportedTranslations}}.

\code{\link[=translations_write]{translations_write()}} returns \code{NULL}, invisibly.

\code{\link[=translations_paths]{translations_paths()}} returns a named character vector.
}
\description{
Export \code{\link{Translator}} objects to text files and import such
files back into \R as \code{\link{Translator}} objects.
}
\details{
The information contained within a \code{\link{Translator}} object is
split: translations are reorganized by language and exported independently
from other fields.

\code{\link[=translator_write]{translator_write()}} creates two types of file: a single \emph{Translator file},
and zero, or more \emph{translations files}. These are plain text files that can
be inspected and modified using a wide variety of tools and systems. They
target different audiences:
\itemize{
\item the Translator file is useful to developers, and
\item translations files are meant to be shared with non-technical
collaborators such as translators.
}

\code{\link[=translator_read]{translator_read()}} first reads a Translator file and creates a
\code{\link{Translator}} object from it. It then calls
\code{\link[=translations_paths]{translations_paths()}} to list expected translations files (that should
normally be stored alongside the Translator file), attempts to read them,
and registers successfully imported translations.

There are two requirements.
\itemize{
\item All files must be stored in the same directory. By default, this is set
equal to \verb{inst/transltr/} (see \code{getOption("transltr.path")}).
\item Filenames of translations files are standardized and must correspond to
languages (language codes, see \code{lang}).
}

The inner workings of the serialization process are thoroughly described in
\code{\link[=serialize]{serialize()}}.
\subsection{Translator file}{

A Translator file contains a \href{https://yaml.org/spec/1.1/}{YAML} (1.1)
representation of a \code{\link{Translator}} object stripped of all
its translations except those that are registered as source text.
}

\subsection{Translations files}{

A translations file contains a \link[=flat_serialize]{FLAT} representation of
a set of translations sharing the same target language. This format attempts
to be as simple as possible for non-technical collaborators.
}
}
\examples{
# Set source language.
language_source_set("en")

# Create a path to a temporary Translator file.
temp_path <- tempfile(pattern = "translator_", fileext = ".yml")
temp_dir  <- dirname(temp_path)  ## tempdir() could also be used

# Create a Translator object.
# This would normally be done by find_source(), or translator_read().
tr <- translator(
  id = "test-translator",
  en = "English",
  es = "Español",
  fr = "Français",
  text(
    en = "Hello, world!",
    fr = "Bonjour, monde!"),
  text(
    en = "Farewell, world!",
    fr = "Au revoir, monde!"))

# Export it. This creates 3 files: 1 Translator file, and 2 translations
# files because two non-source languages are registered. The file for
# language "es" contains placeholders and must be completed.
translator_write(tr, temp_path)
translator_read(temp_path)

# Translations can be read individually.
translations_files <- translations_paths(tr, temp_dir)
translations_read(translations_files[["es"]])
translations_read(translations_files[["fr"]])

# This is rarely useful, but translations can also be exported individually.
# You may use this to add a new language, as long as it has an entry in the
# underlying Translator object (or file).
tr$set_native_languages(el = "Greek")

translations_files <- translations_paths(tr, temp_dir)

translations_write(tr, translations_files[["el"]], "el")
translations_read(file.path(temp_dir, "el.txt"))

}
\seealso{
\code{\link{Translator}},
\code{\link[=serialize]{serialize()}}
}
