% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mixture_identification.R
\encoding{UTF-8}
\name{mixmod_regression.default}
\alias{mixmod_regression.default}
\title{Mixture Model Identification using Segmented Regression}
\usage{
\method{mixmod_regression}{default}(
  x,
  y,
  status,
  distribution = c("weibull", "lognormal", "loglogistic"),
  conf_level = 0.95,
  k = 2,
  control = segmented::seg.control(),
  ...
)
}
\arguments{
\item{x}{A numeric vector which consists of lifetime data. Lifetime data
could be every characteristic influencing the reliability of a product,
e.g. operating time (days/months in service), mileage (km, miles), load
cycles.}

\item{y}{A numeric vector which consists of estimated failure probabilities
regarding the lifetime data in \code{x}.}

\item{status}{A vector of binary data (0 or 1) indicating whether a unit is
a right censored observation (= 0) or a failure (= 1).}

\item{distribution}{Supposed distribution of the random variable.}

\item{conf_level}{Confidence level of the interval.}

\item{k}{Number of mixture components. If the data should be split in an
automated fashion, \code{k} must be set to \code{NULL}. The argument \code{fix.psi} of
\code{control} is then set to \code{FALSE}.}

\item{control}{Output of the call to \link[segmented:seg.control]{seg.control},
which is passed to \link[segmented:segmented]{segmented.lm}. See 'Examples' for usage.}

\item{...}{Further arguments passed to or from other methods. Currently not used.}
}
\value{
A list with classes \code{wt_model} and \code{wt_rank_regression} if no breakpoint
was detected. See \link{rank_regression}. The returned tibble \code{data} is of class
\code{wt_cdf_estimation} and contains the dummy columns \code{cdf_estimation_method} and
\code{id}. The former is filled with \code{NA_character}, due to internal usage and the
latter is filled with \code{"XXXXXX"} to point out that unit identification is not
possible when using the vector-based approach.

A list with classes \code{wt_model} and \code{wt_mixmod_regression} if at least one
breakpoint was determined. The length of the list depends on the number of
identified subgroups. Each list contains the information provided by
\link{rank_regression}. The returned tibble \code{data} of each list element only retains
information on the failed units and has modified and additional columns:
\itemize{
\item \code{id} : Modified id, overwritten with \code{"XXXXXX"} to point out that unit
identification is not possible when using the vector-based approach.
\item \code{cdf_estimation_method} : A character that is always \code{NA_character}. Only
needed for internal use.
\item \code{q} : Quantiles of the standard distribution calculated from column \code{prob}.
\item \code{group} : Membership to the respective segment.
}
}
\description{
This function uses piecewise linear regression to divide the data into
subgroups. See 'Details'.
}
\details{
The segmentation process is based on the lifetime realizations of failed
units and their corresponding estimated failure probabilities for which intact
items are taken into account. It is performed with the support of
\link[segmented:segmented]{segmented.lm}.

Segmentation can be done with a specified number of subgroups or in an automated
fashion (see argument \code{k}). The algorithm tends to overestimate the number of
breakpoints when the separation is done automatically (see 'Warning' in
\link[segmented:segmented]{segmented.lm}).

In the context of reliability analysis it is important that the main types of
failures can be identified and analyzed separately. These are
\itemize{
\item early failures,
\item random failures and
\item wear-out failures.
}

In order to reduce the risk of overestimation as well as being able to consider
the main types of failures, a maximum of three subgroups (\code{k = 3}) is recommended.
}
\examples{
# Vectors:
## Data for mixture model:
hours <- voltage$hours
status <- voltage$status

## Data for simple unimodal distribution:
distance <- shock$distance
status_2 <- shock$status

# Probability estimation with one method:
prob_mix <- estimate_cdf(
  x = hours,
  status = status,
  method = "johnson"
)

prob <- estimate_cdf(
  x = distance,
  status = status_2,
  method = "johnson"
)

 # Example 1 - Mixture identification using k = 2 two-parametric Weibull models:
mix_mod_weibull <- mixmod_regression(
   x = prob_mix$x,
   y = prob_mix$prob,
   status = prob_mix$status,
   distribution = "weibull",
   conf_level = 0.99,
   k = 2
)

# Example 2 - Mixture identification using k = 3 two-parametric lognormal models:
mix_mod_lognorm <- mixmod_regression(
   x = prob_mix$x,
   y = prob_mix$prob,
   status = prob_mix$status,
   distribution = "lognormal",
   k = 3
)

# Example 3 - Mixture identification using control argument:
mix_mod_control <- mixmod_regression(
  x = prob_mix$x,
  y = prob_mix$prob,
  status = prob_mix$status,
  distribution = "weibull",
  k = 2,
  control = segmented::seg.control(display = TRUE)
)

# Example 4 - Mixture identification performs rank_regression for k = 1:
mod <- mixmod_regression(
  x = prob$x,
  y = prob$prob,
  status = prob$status,
  distribution = "weibull",
  k = 1
)

}
\references{
Doganaksoy, N.; Hahn, G.; Meeker, W. Q., Reliability Analysis by
Failure Mode, Quality Progress, 35(6), 47-52, 2002
}
\seealso{
\link{mixmod_regression}
}
