% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/xml_fragment.R
\name{xml_fragment}
\alias{xml_fragment}
\title{Create an XML fragment}
\usage{
xml_fragment(...)
}
\arguments{
\item{...}{nest named elements and characters to include in the fragment (see example)}
}
\value{
an \code{xml_fragment}, list object that can be converted to an \code{xml2::xml_document}
or \code{character} string
}
\description{
Create an XML fragment using readable R syntax, that can be used to create
a string, an \code{xml2::xml_document} or as a building block for more complex XML documents.
}
\details{
An \code{xml_fragment} is built using:
\itemize{
\item named \code{frag} elements, each name is a tag name, and the value is the contents
of the tag, e.g. \code{name = "value"} becomes \verb{<name>value</name>}. The value
can be a nested \code{frag} object, a character string or a numeric value.
\item \code{.attr} attributes, which is set on current element, or on the \code{frag} where
it is specified
\item unnamed elements, which are added as text nodes.
\item \code{\link[=data_frag]{data_frag()}} function that can be used to convert a data.frame to an xml fragment,
in which each row is a set of xml elements (columns).
\item \code{\link[=tag]{tag()}} function that can be used to create a tag with attributes and (optional) text.
}

An \code{xml_doc} is a special case of an \code{xml_fragment} that contains exactly one
root element, and errors when this is not the case.

A resulting \code{xml_fragment} object can be converted to an \code{xml2::xml_document} with
\code{\link[xml2:as_xml_document]{xml2::as_xml_document()}} or to a character string with \code{\link[=as.character]{as.character()}}. Both
methods are fast using a performant c++ implementation.
}
\examples{
doc <- xml_fragment(
  study = frag(
    .attr = c(id="1"),
    person = frag(
      .attr = c(id = "p1"),
      name = "John Doe",
      age = 30
    ),
    person = frag(
      name = "Jane Doe",
      age = 25,
      address = frag(street = "123 Main St", city = "Springfield"),
      "This is a text node"
    )
  )
)

print(doc)
if (require("xml2")){
  as_xml_document(doc)
}

# you can create a function to generate an xml fragment:
person_frag <- function(name, age, id){
  tag("person", id = id) / frag(
    name = name,
    age  = age,
    address = frag(
      street = "123 Main St",
      city = "Springfield"
    )
  )
}

# xml_doc is a xml_fragment with the restriction of having one root element
doc2 <- xml_doc("study") / (
  person_frag("John Doe", 30, "p1") +
  person_frag("Jane Doe", 25, "p2")
)

print(doc2)

if (require("xml2")){
  as_xml_document(doc2)
}

# a fragment can have multiple root elements
fgmt <- person_frag("John Doe", 30, id = "p1") +
  person_frag("Jane Doe", 25, id = "p2")


print(fgmt)

if (require("xml2")){
  # as_xml_document won't work because it expects a single root element,
  # so we retrieve a nodeset instead
  as_xml_nodeset(fgmt)
}

iris_xml <- xml_doc("fieldstudy", id = "iris", doi ="10.1111/j.1469-1809.1936.tb02137.x") /
  frag(
    source = "Fisher, R. A. (1936) The use of multiple measurements in
taxonomic problems. Annals of Eugenics, 7, Part II, 179–188.",
    data = data_frag(iris, row_tag = "obs")
  )

print(iris_xml, max_characters = 300)

if (require("xml2")){
  as_xml_document(iris_xml)
}
}
\seealso{
Other xml_fragment: 
\code{\link{add_child_fragment}()},
\code{\link{as.character.xml_fragment}()},
\code{\link{as_frag}()},
\code{\link{as_xml_nodeset}()},
\code{\link{data_frag}()},
\code{\link{frag}()}
}
\concept{xml_fragment}
