use std::fmt;
use std::iter::{Fuse, FusedIterator, Peekable};






#[must_use = "iterator adaptors are lazy and do nothing unless consumed"]
pub struct WithPosition<I>
where
    I: Iterator,
{
    handled_first: bool,
    peekable: Peekable<Fuse<I>>,
}

impl<I> fmt::Debug for WithPosition<I>
where
    I: Iterator,
    Peekable<Fuse<I>>: fmt::Debug,
{
    debug_fmt_fields!(WithPosition, handled_first, peekable);
}

impl<I> Clone for WithPosition<I>
where
    I: Clone + Iterator,
    I::Item: Clone,
{
    clone_fields!(handled_first, peekable);
}


pub fn with_position<I>(iter: I) -> WithPosition<I>
where
    I: Iterator,
{
    WithPosition {
        handled_first: false,
        peekable: iter.fuse().peekable(),
    }
}





#[derive(Copy, Clone, Debug, PartialEq, Eq)]
pub enum Position {

    First,

    Middle,

    Last,

    Only,
}

impl<I: Iterator> Iterator for WithPosition<I> {
    type Item = (Position, I::Item);

    fn next(&mut self) -> Option<Self::Item> {
        match self.peekable.next() {
            Some(item) => {
                if !self.handled_first {

                    self.handled_first = true;


                    match self.peekable.peek() {
                        Some(_) => Some((Position::First, item)),
                        None => Some((Position::Only, item)),
                    }
                } else {


                    match self.peekable.peek() {
                        Some(_) => Some((Position::Middle, item)),
                        None => Some((Position::Last, item)),
                    }
                }
            }

            None => None,
        }
    }

    fn size_hint(&self) -> (usize, Option<usize>) {
        self.peekable.size_hint()
    }

    fn fold<B, F>(mut self, mut init: B, mut f: F) -> B
    where
        F: FnMut(B, Self::Item) -> B,
    {
        if let Some(mut head) = self.peekable.next() {
            if !self.handled_first {


                match self.peekable.next() {
                    Some(second) => {
                        let first = std::mem::replace(&mut head, second);
                        init = f(init, (Position::First, first));
                    }
                    None => return f(init, (Position::Only, head)),
                }
            }

            init = self.peekable.fold(init, |acc, mut item| {
                std::mem::swap(&mut head, &mut item);
                f(acc, (Position::Middle, item))
            });

            init = f(init, (Position::Last, head));
        }
        init
    }
}

impl<I> ExactSizeIterator for WithPosition<I> where I: ExactSizeIterator {}

impl<I: Iterator> FusedIterator for WithPosition<I> {}
