





use crate::join_context;

use super::IndexedParallelIterator;

use std::usize;







pub trait ProducerCallback<T> {




    type Output;




    fn callback<P>(self, producer: P) -> Self::Output
    where
        P: Producer<Item = T>;
}

























pub trait Producer: Send + Sized {


    type Item;


    type IntoIter: Iterator<Item = Self::Item> + DoubleEndedIterator + ExactSizeIterator;



    fn into_iter(self) -> Self::IntoIter;











    fn min_len(&self) -> usize {
        1
    }










    fn max_len(&self) -> usize {
        usize::MAX
    }



    fn split_at(self, index: usize) -> (Self, Self);





    fn fold_with<F>(self, folder: F) -> F
    where
        F: Folder<Self::Item>,
    {
        folder.consume_iter(self.into_iter())
    }
}















pub trait Consumer<Item>: Send + Sized {

    type Folder: Folder<Item, Result = Self::Result>;


    type Reducer: Reducer<Self::Result>;


    type Result: Send;





    fn split_at(self, index: usize) -> (Self, Self, Self::Reducer);



    fn into_folder(self) -> Self::Folder;



    fn full(&self) -> bool;
}







pub trait Folder<Item>: Sized {

    type Result;


    fn consume(self, item: Item) -> Self;









    fn consume_iter<I>(mut self, iter: I) -> Self
    where
        I: IntoIterator<Item = Item>,
    {
        for item in iter {
            self = self.consume(item);
            if self.full() {
                break;
            }
        }
        self
    }


    fn complete(self) -> Self::Result;



    fn full(&self) -> bool;
}








pub trait Reducer<Result> {


    fn reduce(self, left: Result, right: Result) -> Result;
}






pub trait UnindexedConsumer<I>: Consumer<I> {







    fn split_off_left(&self) -> Self;



    fn to_reducer(&self) -> Self::Reducer;
}









pub trait UnindexedProducer: Send + Sized {

    type Item;


    fn split(self) -> (Self, Option<Self>);



    fn fold_with<F>(self, folder: F) -> F
    where
        F: Folder<Self::Item>;
}






#[derive(Clone, Copy)]
struct Splitter {



    splits: usize,
}

impl Splitter {
    #[inline]
    fn new() -> Splitter {
        Splitter {
            splits: crate::current_num_threads(),
        }
    }

    #[inline]
    fn try_split(&mut self, stolen: bool) -> bool {
        let Splitter { splits } = *self;

        if stolen {


            self.splits = Ord::max(crate::current_num_threads(), self.splits / 2);
            true
        } else if splits > 0 {

            self.splits /= 2;
            true
        } else {

            false
        }
    }
}



#[derive(Clone, Copy)]
struct LengthSplitter {
    inner: Splitter,



    min: usize,
}

impl LengthSplitter {









    #[inline]
    fn new(min: usize, max: usize, len: usize) -> LengthSplitter {
        let mut splitter = LengthSplitter {
            inner: Splitter::new(),
            min: Ord::max(min, 1),
        };





        let min_splits = len / Ord::max(max, 1);


        if min_splits > splitter.inner.splits {
            splitter.inner.splits = min_splits;
        }

        splitter
    }

    #[inline]
    fn try_split(&mut self, len: usize, stolen: bool) -> bool {

        len / 2 >= self.min && self.inner.try_split(stolen)
    }
}












pub fn bridge<I, C>(par_iter: I, consumer: C) -> C::Result
where
    I: IndexedParallelIterator,
    C: Consumer<I::Item>,
{
    let len = par_iter.len();
    return par_iter.with_producer(Callback { len, consumer });

    struct Callback<C> {
        len: usize,
        consumer: C,
    }

    impl<C, I> ProducerCallback<I> for Callback<C>
    where
        C: Consumer<I>,
    {
        type Output = C::Result;
        fn callback<P>(self, producer: P) -> C::Result
        where
            P: Producer<Item = I>,
        {
            bridge_producer_consumer(self.len, producer, self.consumer)
        }
    }
}














pub fn bridge_producer_consumer<P, C>(len: usize, producer: P, consumer: C) -> C::Result
where
    P: Producer,
    C: Consumer<P::Item>,
{
    let splitter = LengthSplitter::new(producer.min_len(), producer.max_len(), len);
    return helper(len, false, splitter, producer, consumer);

    fn helper<P, C>(
        len: usize,
        migrated: bool,
        mut splitter: LengthSplitter,
        producer: P,
        consumer: C,
    ) -> C::Result
    where
        P: Producer,
        C: Consumer<P::Item>,
    {
        if consumer.full() {
            consumer.into_folder().complete()
        } else if splitter.try_split(len, migrated) {
            let mid = len / 2;
            let (left_producer, right_producer) = producer.split_at(mid);
            let (left_consumer, right_consumer, reducer) = consumer.split_at(mid);
            let (left_result, right_result) = join_context(
                |context| {
                    helper(
                        mid,
                        context.migrated(),
                        splitter,
                        left_producer,
                        left_consumer,
                    )
                },
                |context| {
                    helper(
                        len - mid,
                        context.migrated(),
                        splitter,
                        right_producer,
                        right_consumer,
                    )
                },
            );
            reducer.reduce(left_result, right_result)
        } else {
            producer.fold_with(consumer.into_folder()).complete()
        }
    }
}




pub fn bridge_unindexed<P, C>(producer: P, consumer: C) -> C::Result
where
    P: UnindexedProducer,
    C: UnindexedConsumer<P::Item>,
{
    let splitter = Splitter::new();
    bridge_unindexed_producer_consumer(false, splitter, producer, consumer)
}

fn bridge_unindexed_producer_consumer<P, C>(
    migrated: bool,
    mut splitter: Splitter,
    producer: P,
    consumer: C,
) -> C::Result
where
    P: UnindexedProducer,
    C: UnindexedConsumer<P::Item>,
{
    if consumer.full() {
        consumer.into_folder().complete()
    } else if splitter.try_split(migrated) {
        match producer.split() {
            (left_producer, Some(right_producer)) => {
                let (reducer, left_consumer, right_consumer) =
                    (consumer.to_reducer(), consumer.split_off_left(), consumer);
                let bridge = bridge_unindexed_producer_consumer;
                let (left_result, right_result) = join_context(
                    |context| bridge(context.migrated(), splitter, left_producer, left_consumer),
                    |context| bridge(context.migrated(), splitter, right_producer, right_consumer),
                );
                reducer.reduce(left_result, right_result)
            }
            (producer, None) => producer.fold_with(consumer.into_folder()).complete(),
        }
    } else {
        producer.fold_with(consumer.into_folder()).complete()
    }
}
