\name{MixNRMI2cens}

\alias{MixNRMI2cens}

\title{Normalized Random Measures Mixture of Type II for censored data}

\description{
    Bayesian nonparametric estimation based on normalized measures driven mixtures
    for locations and scales.
}

\usage{
MixNRMI2cens(xleft, xright, probs = c(0.025, 0.5, 0.975), Alpha = 1, 
    Beta = 0, Gama = 0.4, distr.k = 1, distr.py0 = 1, distr.pz0 = 2, 
    mu.pz0 = 3, sigma.pz0 = sqrt(10), delta = 4, kappa = 2, Delta = 2, 
    Meps = 0.01, Nx = 150, Nit = 1500, Pbi = 0.1, epsilon = NULL, 
    printtime = TRUE, extras = FALSE) 
}

\arguments{
  \item{xleft}{Numeric vector. Lower limit of interval censoring. For exact data the same as xright}
  \item{xright}{Numeric vector. Upper limit of interval censoring. For exact data the same as xleft.}
  \item{probs}{Numeric vector. Desired quantiles of the density estimates.}
  \item{Alpha}{Numeric constant. Total mass of the centering measure.
        See details.}
  \item{Beta}{Numeric positive constant. See details.}
  \item{Gama}{Numeric constant. \eqn{0 \leq Gama \leq 1}{0 <= Gama <=1}.
            See details.}
  \item{distr.k}{Integer number identifying the mixture kernel: 1 = Normal;
            2 = Gamma; 3 = Beta; 4 = Double Exponential; 5 = Lognormal.}
  \item{distr.py0}{Integer number identifying the centering measure for
            locations: 1 = Normal; 2 = Gamma; 3 = Beta.}
  \item{distr.pz0}{Integer number identifying the centering measure for scales:
        2 = Gamma, 5 = Lognormal, 6 = Half Cauchy, 7 = Half Normal, 8 = Half Student-t, 9 = Uniform, 10 = Truncated Normal.}
  \item{mu.pz0}{Numeric constant. Prior mean of the centering measure for scales.}
  \item{sigma.pz0}{Numeric constant. Prior standard deviation of the
            centering measure for scales.}
  \item{delta}{Numeric positive constant. Metropolis-Hastings proposal
            variation coefficient for sampling the scales.}
  \item{kappa}{Numeric positive constant. Metropolis-Hastings proposal
            variation coefficient for sampling the location parameters.}
  \item{Delta}{Numeric positive constant. Metropolis-Hastings proposal
        variation coefficient for sampling the latent U.}
  \item{Meps}{Numeric constant. Relative error of the jump sizes in the continuous
    component of the process. Smaller values imply larger number of jumps.}
  \item{Nx}{Integer constant. Number of grid points for the evaluation of the
        density estimate.}
  \item{Nit}{Integer constant. Number of MCMC iterations.}
  \item{Pbi}{Numeric constant. Burn-in period proportion of \code{Nit}.}
  \item{epsilon}{Numeric constant. Extension to the evaluation grid range.
        See details.}
  \item{printtime}{Logical. If TRUE, prints out the execution time.}
  \item{extras}{Logical. If TRUE, gives additional objects: means, sigmas, weights and Js.}
}
\details{
This generic function fits a normalized random measure (NRMI) mixture model for
density estimation (James et al. 2009). Specifically, the model assumes a
normalized generalized gamma (NGG) prior for both, locations (means) and
standard deviations, of the mixture kernel, leading to a fully nonparametric
mixture model.

The details of the model are:
\deqn{X_i|Y_i,Z_i \sim k(\cdot|Y_i,Z_i)}{X_i|Y_i,Z_i ~ k(.|Y_i,Z_i)}
\deqn{(Y_i,Z_i)|P \sim P, i=1,\dots,n}{(Y_i,Z_i)|P ~ P,  i=1,...,n}
\deqn{P \sim \textrm{NGG}(\texttt{Alpha, Beta, Gama; P\_0})}{P ~ NGG(Alpha, Beta, Gama; P_0)}
where, \eqn{X_i}'s are the observed data, \eqn{(Y_i,Z_i)}'s are bivariate
latent (location and scale) vectors, \code{k} is a parametric kernel parameterized
in  terms of mean and standard deviation, \code{(Alpha, Beta, Gama; P_0)} are the
parameters of the NGG prior with a bivariate \code{P_0} being the centering measure with independent components, that is, \eqn{P_0(Y,Z) = P_0(Y)*P_0(Z)}. The parameters of \code{P_0(Y)} are assigned vague hyper prior distributions and \code{(mu.pz0,sigma.pz0)} are the hyper-parameters of \code{P_0(Z)}. In particular, \code{NGG(Alpha, 1, 0; P_0)} defines a Dirichlet process; \code{NGG(1, Beta, 1/2;P_0)} defines a Normalized inverse Gaussian process; and \code{NGG(1, 0, Gama; P_0)} defines a normalized stable process. The evaluation grid ranges from \code{min(x) - epsilon} to \code{max(x) + epsilon}.
By default \code{epsilon=sd(x)/4}.
}

\value{
    The function returns a list with the following components:
    \item{xx}{Numeric vector. Evaluation grid.}
    \item{qx}{Numeric array. Matrix of dimension
        \eqn{\texttt{Nx} \times (\texttt{length(probs)} + 1)}{Nx x (length(probs)+1)}
        with the posterior mean and the desired quantiles input in \code{probs}.}
    \item{cpo}{Numeric vector of \code{length(x)} with conditional predictive
        ordinates.}
    \item{R}{Numeric vector of \code{length(Nit*(1-Pbi))} with the number of
        mixtures components (clusters).}
    \item{U}{Numeric vector of \code{length(Nit*(1-Pbi))} with the values of the
        latent variable U.}
    \item{Allocs}{List of \code{length(Nit*(1-Pbi))} with the clustering allocations.}
    \item{means}{List of \code{length(Nit*(1-Pbi))} with the cluster means (locations). Only if extras = TRUE.}
    \item{sigmas}{Numeric vector of \code{length(Nit*(1-Pbi))} with the cluster standard deviations. Only if extras = TRUE.}
    \item{weights}{List of \code{length(Nit*(1-Pbi))} with the mixture weights. Only if extras = TRUE.}
    \item{Js}{List of \code{length(Nit*(1-Pbi))} with the unnormalized weights (jump sizes). Only if extras = TRUE.}
    \item{Nm}{Integer constant. Number of jumps of the continuous component of the 
            unnormalized process.}
    \item{Nx}{Integer constant. Number of grid points for the evaluation of the
        density estimate.}
    \item{Nit}{Integer constant. Number of MCMC iterations.}
    \item{Pbi}{Numeric constant. Burn-in period proportion of \code{Nit}.}
    \item{procTime}{Numeric vector with execution time provided by
        \code{proc.time} function.}
}

\references{
1.- Barrios, E., Lijoi, A., Nieto-Barajas, L. E. and Prüenster, I. (2013). Modeling with Normalized Random Measure Mixture Models. Statistical Science. Vol. 28, No. 3, 313-334.

2.- James, L.F., Lijoi, A. and Prüenster, I. (2009). Posterior analysis for
normalized random measure with independent increments. Scand. J. Statist 36,
76-97.

3.- Kon Kam King, G., Arbel, J. and Prüenster, I. (2016). Species Sensitivity Distribution revisited: a Bayesian nonparametric approach. In preparation.
}

\author{Barrios, E., Kon Kam King, G. and Nieto-Barajas, L.E.}

\section{Warning }{The function is computing intensive. Be patient.}

\seealso{\code{\link{MixNRMI2}}, \code{\link{MixNRMI1}}, \code{\link{MixNRMI1cens}}}

\examples{
\dontrun{
### Example 1
# Data
data(acidity)
x <- acidity
# Fitting the model under default specifications
out <- MixNRMI2cens(x,x)
# Plotting density estimate + 95% credible interval
attach(out)
m <- ncol(qx)
ymax <- max(qx[,m])
par(mfrow=c(1,1))
hist(x,probability=TRUE,breaks=20,col=grey(.9),ylim=c(0,ymax))
lines(xx,qx[,1],lwd=2)
lines(xx,qx[,2],lty=3,col=4)
lines(xx,qx[,m],lty=3,col=4)
detach()
}

\dontrun{
### Example 2
# Data
data(salinity)
# Fitting the model under special specifications
out <- MixNRMI2cens(xleft=salinity$left,xright=salinity$right,Nit=5000,distr.pz0=10,
  mu.pz0=1,sigma.pz0=2)
# Plotting density estimate + 95% credible interval
attach(out)
m <- ncol(qx)
ymax <- max(qx[,m])
par(mfrow=c(1,1))
plot(xx,qx$"q0.5",lwd=2,type="l",ylab="Density",xlab="Data")
lines(xx,qx[,2],lty=3,col=4)
lines(xx,qx[,m],lty=3,col=4)
# Plotting number of clusters
par(mfrow=c(2,1))
plot(R,type="l",main="Trace of R")
hist(R,breaks=min(R-0.5):max(R+0.5),probability=TRUE)
detach()
}
}

\keyword{distribution}
\keyword{models}
\keyword{nonparametrics}
