% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get.oc.comb.R
\name{get.oc.comb}
\alias{get.oc.comb}
\title{Generate operating characteristics for drug combination trials}
\usage{
get.oc.comb(target, p.true, ncohort, cohortsize, n.earlystop=NULL, startdose=c(1, 1),
                   titration=FALSE,p.saf=0.6*target, p.tox=1.4*target, cutoff.eli=0.95,
                   extrasafe=FALSE,offset=0.05, ntrial=1000, mtd.contour=FALSE,
                   boundMTD=FALSE, seed=6)
}
\arguments{
\item{target}{the target DLT rate}

\item{p.true}{a \code{J*K} matrix \code{(J<=K)} containing the true toxicity probabilities of
combinations with \code{J} dose levels of agent A and \code{K} dose levels of agent B}

\item{ncohort}{a \code{1*J} vector specifying the number of cohorts for each of \code{J} subtrials
if \code{mtd.contour=TRUE}; Otherwise, a scalar specifying the total number of cohorts for
the trial.}

\item{cohortsize}{the cohort size}

\item{n.earlystop}{the early stopping parameter. If the number of patients treated at the current
dose reaches \code{n.earlystop}, stop the trial or subtrial and select the MTD based on
the observed data. When the waterfall design is used to find the MTD contour,
\code{n.earlystop=12} by default.}

\item{startdose}{the starting dose combination level for drug combination trial}

\item{titration}{set \code{titration=TRUE} to perform dose escalation with cohort size = 1 to accelerate dose escalation at the begining of the trial.}

\item{p.saf}{the highest toxicity probability that is deemed subtherapeutic (i.e. below the MTD)
such that dose escalation should be undertaken.
The default value is \code{p.saf=0.6*target}.}

\item{p.tox}{the lowest toxicity probability that is deemed overly toxic such that deescalation
is required. The default value is \code{p.tox=1.4*target}.}

\item{cutoff.eli}{the cutoff to eliminate an overly toxic dose for safety. We recommend the
default value of (\code{cutoff.eli=0.95}) for general use.}

\item{extrasafe}{set \code{extrasafe=TRUE} to impose a more stringent stopping rule}

\item{offset}{a small positive number (between 0 and 0.5) to control how strict the stopping
rule is when \code{extrasafe=TRUE}. A larger value leads to a more strict stopping
rule. The default value \code{offset=0.05} generally works well.}

\item{ntrial}{the total number of trials to be simulated}

\item{mtd.contour}{set \code{mtd.contour=TRUE} to select the MTD contour (claiming multiple MTDs).
Otherwise, BOIN design is used to search for a single MTD.}

\item{boundMTD}{set \code{boundMTD=TRUE} to impose the condition: the isotonic estimate of toxicity
probability for the selected MTD must be less than de-escalation boundary.}

\item{seed}{the random seed for simulation}
}
\value{
\code{get.oc.comb()} returns the operating characteristics of the BOIN combination or
waterfall design as a list. For the BOIN combination design, including:
(1) true toxicity probability at each dose level (\code{$p.true}),
(2) selection percentage at each dose level (\code{$selpercent}),
(3) the number of patients treated at each dose level (\code{$npatients})
(4) the number of toxicities observed at each dose level (\code{$ntox}),
(5) the total number of patients in the trial (\code{$totaln}),
(6) the total number of toxicities observed for the trial (\code{$totaltox})
(7) the pecentage of correct selection (\code{$pcs}),
(8) the total percentage of patients treated at the MTD (\code{$npercent}).
(9) the percentage of early stopping without selecting the MTD (\code{$percentstop})
For the the waterfall design, including:
(1) true toxicity probability at each dose level (\code{$p.true}),
(2) selection percentage of dose combinations (\code{$selpercent}),
(3) the number of patients treated at each dose combination (\code{$npatients})
(4) the number of toxicities observed at each dose combination (\code{$ntox}),
(5) the total number of patients in the trial (\code{$totaln}),
(6) the total number of toxicities observed for the trial (\code{$totaltox})
(7) the total percentage of correct selection at the MTD contour (\code{$pcs.contour}),
(8) the total percentage of patients treated at MTD contour
(\code{$npercent.contour})
(9) the total percentage of patients treated above MTD contour
(\code{$npercent.above.contour})
(10) the total percentage of patients treated below MTD contour
(\code{$npercent.below.contour})
}
\description{
Obtain the operating characteristics of the BOIN design or waterfall design for drug combination
 trials. The BOIN design is to find a MTD, and the waterfall design is to find the MTD contour
 (i.e., multple MTDs in the dose matrix)
}
\details{
The operating characteristics of the BOIN design or waterfall design are generated by
simulating trials under the prespecified true toxicity probabilities of the investigational dose
combinations. If \code{titration=TRUE}, we perform dose escalation with cohort size = 1 at the begining of the trial:
starting from \code{startdose}, if no toxicity is observed, we escalate the dose;
otherwise, the titration is completed and we switch to cohort size = \code{cohortsize}.
Titration accelerates the dose escalation and is useful when low doses are believed to be safe.


The BOIN and waterfall designs have two built-in stopping rules:
(1) stop the trial/subtrial if the lowest dose is eliminated due to toxicity, and no dose should
be selected as the MTD; and (2) stop the trial/subtrial and select the MTD if the number of
patients treated at the current dose reaches \code{n.earlystop}. The first stopping rule is a safety
rule to protect patients from the case in which all doses are overly toxic. The rationale for
the second stopping rule is that when there is a large number (i.e., \code{n.earlystop}) of
patients assigned to a dose, it means that the dose-finding algorithm has approximately converged.
Thus, we can stop the trial/subtrial early and select the MTD to save sample size and reduce the
trial duration.


 For some applications, investigators may prefer a more strict safety stopping rule than rule
 (1) for extra safety when the lowest dose is overly toxic.
 This can be achieved by setting \code{extrasafe=TRUE},
 which imposes the following more strict safety stopping rule:
 stop the trial if (i) the number of patients treated at the lowest dose \code{>=3},
and (ii) \eqn{Pr(toxicity\ rate\ of\ the\ lowest\ dose > \code{target} | data) > \code{cutoff.eli}-\code{offset}}.
As a tradeoff, the strong stopping rule will decrease the MTD selection percentage
 when the lowest dose actually is the MTD.
}
\note{
We should avoid setting the values of \code{p.saf} and \code{p.tox} very close to the
     \code{target}. This is because the small sample sizes of typical phase I trials prevent us from
      differentiating the target DLT rate from the rates close to it. The default values provided by
      \code{get.oc()} are strongly recommended, and generally yield excellent operating characteristics.
}
\examples{

###### drug-combination trial ######

##### combination trial to find a single MTD  ######

## get the operating characteristics for BOIN design
p.true <- matrix(c(0.01,0.03,0.10,0.20,0.30,
                 0.03,0.05,0.15,0.30,0.60,
                 0.08,0.10,0.30,0.60,0.75), byrow=TRUE, ncol=5)

oc.comb <- get.oc.comb(target=0.3, p.true, ncohort=20, cohortsize=3,
   n.earlystop=12, startdose=c(1,1), ntrial=100)

summary(oc.comb)
plot(oc.comb)


## get the operating characteristics with titration for BOIN design
oc.comb <- get.oc.comb(target=0.3, p.true, ncohort=20, cohortsize=3,
   n.earlystop=12, startdose=c(1,1), titration=TRUE, ntrial=100)
summary(oc.comb)
plot(oc.comb)


##### combination trial to find the MTD contour ######

## find the MTD contour using waterfall design
oc.comb <- get.oc.comb(target=0.3, p.true, ncohort=c(10,5,5), cohortsize=3,
   n.earlystop=12, startdose=c(1,1), ntrial=100, mtd.contour=TRUE)

summary(oc.comb)
plot(oc.comb)

}
\references{
Liu S. and Yuan, Y. (2015). Bayesian Optimal Interval Designs for Phase I Clinical
            Trials, \emph{Journal of the Royal Statistical Society: Series C}, 64, 507-523.

           Lin R. and Yin, G. (2017). Bayesian Optimal Interval Designs for Dose Finding in
           Drug-combination Trials, \emph{Statistical Methods in Medical Research}, 26, 2155-2167.

           Yan, F., Zhang, L., Zhou, Y., Pan, H., Liu, S. and Yuan, Y. (2020).BOIN: An R Package
           for Designing Single-Agent and Drug-Combination Dose-Finding Trials Using Bayesian Optimal
           Interval Designs. \emph{Journal of Statistical Software}, 94(13),1-32.<doi:10.18637/jss.v094.i13>.


           Zhang L. and Yuan, Y. (2016). A Simple Bayesian Design to Identify the Maximum
           Tolerated Dose Contour for Drug Combination Trials, \emph{Statistics in Medicine}, 35, 4924-4936.
}
\seealso{
Tutorial: \url{http://odin.mdacc.tmc.edu/~yyuan/Software/BOIN/BOIN2.6_tutorial.pdf}

          Paper: \url{http://odin.mdacc.tmc.edu/~yyuan/Software/BOIN/paper.pdf}
}
\author{
Suyu Liu, Liangcai Zhang, Yanhong Zhou, and Ying Yuan
}
