% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hp.R
\name{hp}
\alias{hp}
\title{Bayesian HP}
\usage{
hp(x, Ex, Dx, model = c("binomial", "lognormal", "poisson"),
 M = 50000, bn = round(M/2), thin = 10, m = rep(NA, 8), v = rep(NA, 8),
 inits = NULL, K = NULL, sigma2 = NULL, prop.control = NULL,
 reduced_model = FALSE)
}
\arguments{
\item{x}{Numeric vector of the ages.}

\item{Ex}{Numeric vector with the exposure by age.}

\item{Dx}{Numeric vector with the death counts by age.}

\item{model}{Character string specifying the model to be adopted. The options are: "binomial", "lognormal" or "poisson".}

\item{M}{Positive integer indicating the number of iterations of the MCMC run.}

\item{bn}{Non-negative integer indicating the number of iteration to be discarded as the burn-in period.}

\item{thin}{Positive integer specifying the period for saving samples.}

\item{m}{Numeric vector with the mean of the prior distributions for (A, B, C, D, E, F, G, H).}

\item{v}{Numeric vector with the variance of the prior distributions for (A, B, C, D, E, F, G, H).}

\item{inits}{Numeric vector with the initial values for the parameters (A, B, C, D, E, F, G, H).}

\item{K}{Number that specifies the extra parameter 'K' value for binomial and poisson models. It is considered only if model = "binomial" or model = "poisson". The default value is 1.}

\item{sigma2}{Positive number that specifies initial value of sigma2 in lognormal model. It is considered only if model = "lognormal".}

\item{prop.control}{Positive number that is considered for tuning the acceptance rate of MCMC.}

\item{reduced_model}{Logical value which determines if reduced model should be addressed. If 'TRUE' (default), the first term of the HP curve (infant mortality) is not considered.}
}
\value{
A \code{HP} class object.
\item{summary}{A table with summaries measures of the parameters.}
\item{post.samples}{A list with the chains generated by MCMC.}
\item{data}{A table with the data considered in fitted model.}
\item{info}{A list with some informations of the fitted model like prior distributions mean and variance, initial values.}
}
\description{
This function fits the Heligman-Pollard (HP) model following a Bayesian framework using
Markov chain Monte Carlo techniques to sample the posterior distribution.
Three model options are available: The Binomial and the Poisson models consider nine parameters,
whereas the Log-Normal model considers eight parameters for modelling the HP curve.
}
\details{
The binomial model assumes that Dx, the death count for the age x, has a binomial distribution:
Bin(Ex, qx), where qx is probability of death in age x. The poisson model assumes that Dx has a
Poisson distribution: Po(Ex*qx). Both models consider the nine parameters
HP curve, that was proposed by Heligman and Pollard (1980):

\eqn{HP_x = A^{(x+B)^C} + De^{(-E(log(x/F))^2)} + GH^x/(1+KGH^x)}

\eqn{qx = 1-e^{(-HP_x)}}

This approximation ensures that qx, which is a probability, is in the correct range.

The Log-Normal model assumes that the log odds of death qx/(1-qx) has Normal distribution
with a constant variance for all the ages. This model was proposed by Dellaportas et al.(2001)
and they consider the eighth parameters HP curve as follows:

\eqn{log(qx/(1-qx)) = log(A^{(x+B)^C} + De^{-E(log(x/F))^2} + GH^x) + \epsilon_x},

where \eqn{\epsilon_x} has independent distributions Normal(0, sigma2) for all ages. More details
of this model are available in Dellaportas, Smith e Stavropoulos (2001).

The reduced model does not consider the first term of the HP curve: \eqn{A^{(x+B)^C}}. In this
case, A, B and C are fixed as zero.

All parameters, with the exception of the extra parameter K of the Binomial and the Poisson models
that is fixed in value 1, are estimated by the MCMC methods. Gibbs sampling for sigma2 and
Metropolis-Hastings for parameters A, B, C, D, E, F, G and H. Informative prior distributions
should help the method to converge quickly.
}
\examples{
## Importing mortality data from the USA available on the Human Mortality Database (HMD):
data(USA)

## Selecting the exposure and death count of the 2010 male population ranging from 0 to 90 years old
USA2010 = USA[USA$Year == 2010,]
x = 0:90
Ex = USA2010$Ex.Male[x+1]
Dx = USA2010$Dx.Male[x+1]

## Fitting binomial model
\donttest{fit = hp(x = x, Ex = Ex, Dx = Dx)
print(fit)
fit$summary
fit$info
}
## Fitting lognormal model
## Specifying number of iterations, burn-in, thinning and the initial value of sigma2
fit = hp(x = x, Ex = Ex, Dx = Dx, model = "lognormal",
         M = 1000, bn = 0, thin = 10, sigma2 = 0.1)
summary(fit)

## Fitting poisson model
## Specifying the prior distribution parameters for B and E and fixing K as 0.
\donttest{fit = hp(x = x, Ex = Ex, Dx = Dx, model = "poisson",
         m = c(NA, 0.08, NA, NA, 9, NA, NA, NA),
         v = c(NA, 1e-4, NA, NA, 0.1, NA, NA, NA), K = 0)
summary(fit)
}

## Using other functions available in the package:
## plotting (See "?plot.HP" in the BayesMortalityPlus package for more options):
plot(fit)

## qx estimation (See "?fitted.HP" in the BayesMortalityPlus package for more options):
fitted(fit)

## chain's plot (See "?plot_chain" for more options):
plot_chain(fit)

## credible intervals (See "?qx_ci" for more options):
qx_ci(fit)

}
\references{
Dellaportas, P., Smith, A. F., and Stavropoulos, P. (2001). “Bayesian Analysis of Mortality Data.” \emph{Journal of the Royal Statistical Society: Series A (Statistics in Society)} 164 (2). Wiley Online Library: 275–291.

Heligman, Larry, and John H Pollard. (1980). “The Age Pattern of Mortality.” \emph{Journal of the Institute of Actuaries (1886-1994)} 107 (1). JSTOR: 49–80.
}
\seealso{
\code{\link[=fitted.HP]{fitted.HP()}}, \code{\link[=plot.HP]{plot.HP()}}, \code{\link[=print.HP]{print.HP()}} and \code{\link[=summary.HP]{summary.HP()}} for \code{HP} methods to native R functions \code{\link[=fitted]{fitted()}},
\code{\link[=plot]{plot()}}, \code{\link[=print]{print()}} and \code{\link[=summary]{summary()}}.

\code{\link[=expectancy.HP]{expectancy.HP()}} and \code{\link[=Heatmap.HP]{Heatmap.HP()}} for \code{HP} methods to compute and visualise the truncated life expectancy
via \code{\link[=expectancy]{expectancy()}} and \code{\link[=Heatmap]{Heatmap()}} functions.

\code{\link[=hp_close]{hp_close()}} for close methods to expand the life tables and \code{\link[=hp_mix]{hp_mix()}} for mixing life tables.

\code{\link[=qx_ci]{qx_ci()}} and \code{\link[=plot_chain]{plot_chain()}} to compute credible intervals and visualise the markov chains, respectively.
}
