% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modelRBFN.R
\name{modelRBFN}
\alias{modelRBFN}
\title{RBFN Model}
\usage{
modelRBFN(x, y, distanceFunction, control = list())
}
\arguments{
\item{x}{list of samples in input space}

\item{y}{column vector of observations for each sample}

\item{distanceFunction}{a suitable distance function of type f(x1,x2), returning a scalar distance value, preferably between 0 and 1.
Maximum distances larger 1 are no problem, but may yield scaling bias when different measures are compared.
   Should be non-negative and symmetric.}

\item{control}{(list), with the options for the model building procedure:
\describe{
\item{\code{beta}}{ Parameter of the radial basis function: exp(-beta*D), where D is the distance matrix. If beta is not specified, the heuristic in fbeta will be used to determine it, which is default behavior.}
\item{\code{fbeta}}{ Function f(x) to calculate the beta parameter, x is the maximum distance observed in the input data. Default function is \code{1/(2*(x^2))}.} 
\item{\code{distances}}{ a distance matrix. If available, this matrix is used for model building, instead of calculating the distance matrix using the parameters \code{distanceFunction}. Default is \code{NULL}.}
}}
}
\value{
a fit (list, modelRBFN), with the options and found parameters for the model which has to be passed to the predictor function:
\describe{
\item{\code{SSQ}}{ Variance of the observations (y)}
\item{\code{centers}}{ Centers of the RBFN model, samples in input space (see parameters)}
\item{\code{w}}{ Model parameters (weights) w}
\item{\code{Phi}}{ Gram matrix}
\item{\code{Phinv}}{ (Pseudo)-Inverse of Gram matrix}
\item{\code{w0}}{ Mean of observations (y)}
\item{\code{dMax}}{ Maximum observed distance}
\item{\code{D}}{ Matrix of distances between all samples}
\item{\code{beta}}{ See parameters}
\item{\code{fbeta}}{ See parameters}
\item{\code{distanceFunction}}{See parameters}
}
}
\description{
Implementation of a distance-based Radial Basis Function Network (RBFN) model, e.g., for mixed or combinatorial input spaces.
It is based on employing suitable distance measures for the samples in input space. For reference, see
the paper by Moraglio and Kattan (2011).
}
\examples{
#set random number generator seed
set.seed(1)
#simple test landscape
fn <- landscapeGeneratorUNI(1:5,distancePermutationHamming)
#generate data for training and test
x <- unique(replicate(40,sample(5),FALSE))
xtest <- x[-(1:15)]
x <- x[1:15]
#determin true objective function values
y <- fn(x)
ytest <- fn(xtest)
#build model
fit <- modelRBFN(x,y,distancePermutationHamming)
#predicted obj. function values
ypred <- predict(fit,xtest)$y
#plot
plot(ytest,ypred,xlab="true value",ylab="predicted value",
    pch=20,xlim=c(0.3,1),ylim=c(min(ypred)-0.1,max(ypred)+0.1))
abline(0,1,lty=2)
}
\references{
Moraglio, Alberto, and Ahmed Kattan. "Geometric generalisation of surrogate model based optimisation to combinatorial spaces." Evolutionary Computation in Combinatorial Optimization. Springer Berlin Heidelberg, 2011. 142-154.
}
\seealso{
\code{\link{predict.modelRBFN}}
}
