#' @import RBGL gRbase 

setGeneric("computeDescriptor", function(object,...) {standardGeneric("computeDescriptor")})


#' An S4 class to store the data related to a DAG 
#' @slot network a graph generated by the D2C.network class
#' @slot data  N samples computed according to the network distribution  
#' @slot edges : a set of edges we want to predict
#' @slot labelEdge the label of edges. 1 if there is a direct connection 0 otherwise 
#' @slot X_descriptor the descriptor of the edges computed by the D2C algorithm.
setClass("D2C.Data",  slots = list(network="graph" , data = "matrix" , edges="matrix",labelEdge ="numeric",X_descriptor = "matrix"  ))






 
#' create a data structure that store a network and a sample of the distribution 
#' @param .Object a D2C.data
#' @param network : a network generated by D2C.network
#' @param  ratioMissingNode : numeric a ratio of missing edge  
#' @param nConnected  :  number of connected edge in the training or test sample  
#' @param nDisonnected  :  number of disconnected in the training or test sample  
#' @param data : matrix generated with method compute of D2C.network
#' @param lin : TRUE or FALSE
#' @param functionDescriptor : a descriptor function used in the D2C algorithm 
#' @return N*nNodes matrix
#' @examples
#' require(D2C)
#' require(gRbase)
#' require(RBGL)
#' H=function() function(x) return(x)
#' sdn=0.5
#' sigma=function(x) return(rnorm(n = 1,sd = sdn))
#' DAG = new("D2C.network",network=random_dag(1:50,maxpar = 5,wgt=0.8),H=H,sdn=sdn,sigma=sigma)
#' X = compute(DAG,N=200)
#' DATA = new("D2C.Data",network=DAG@@network,data=X,ratioMissingNode=0.05,
#'            lin=TRUE,nConnected = 4,nDisonnected = 6,functionDescriptor=descriptor)
#' @references Gianluca Bontempi, Maxime Flauder (2014) From dependency to causality: a machine learning approach. Under submission
setMethod("initialize", signature="D2C.Data",function(.Object, network, data ,lin=stop("lin must be specified"),ratioMissingNode = 0.05 ,nConnected = 4,nDisonnected = 6,
                                                      functionDescriptor=stop("functionDescriptor must be specified"))
{
  Nodes = nodes(network)  
  keepNode = sample(Nodes,size = ceiling(length(Nodes)*(1-ratioMissingNode)),replace = F)
  
  #we keep  keepNode in the network and in the data 
  .Object@data = data[,keepNode]  
  .Object@network =subGraph(keepNode, network)
  
  
  ##choose wich edge to train / predict and find the right label
  nEdge = length(edgeList(.Object@network))
  .Object@edges = matrix(unlist(sample(edgeList(.Object@network),size = min(nEdge,nConnected),replace = F)),ncol=2,byrow = TRUE)  
  .Object@edges = rbind(.Object@edges,t(replicate(n =nDisonnected ,sample(keepNode,size=2,replace = FALSE))))
  
  nEdges =  nConnected +nDisonnected
  .Object@labelEdge = numeric(nEdges)
  for(i in 1:nEdges)
  {
    I =.Object@edges[i,1] ; J =.Object@edges[i,2] ;
    .Object@labelEdge[i] = as.numeric(I %in% inEdges(node = J,.Object@network)[[1]])
  } 
  
  
  #compute the descriptor for the edges 
  nNodes = length(.Object@labelEdge)
  X.out = NULL
  for(i in 1:nNodes)
  {
    I =.Object@edges[i,1] ; J =.Object@edges[i,2] ; 
    X.out = rbind(X.out,functionDescriptor(.Object@data,I,J,lin=lin))
  }
  
  .Object@X_descriptor = X.out
  
  
  
  return(.Object)
})




#' computeDescriptor
#' @export
#' @param object a D2C.data object
#' @param lin : TRUE or FALSE
#' @param functionDescriptor : a descriptor function used in the D2C algorithm 
#' @return N*nNodes matrix
setMethod("computeDescriptor", signature="D2C.Data",function(object, functionDescriptor,lin=TRUE)
{
  return(object@X_descriptor)
})

