# Copyright (c) 2022, Adrian Dusa
# All rights reserved.
# 
# Redistribution and use in source and binary forms, with or without
# modification, in whole or in part, are permitted provided that the
# following conditions are met:
#     * Redistributions of source code must retain the above copyright
#       notice, this list of conditions and the following disclaimer.
#     * Redistributions in binary form must reproduce the above copyright
#       notice, this list of conditions and the following disclaimer in the
#       documentation and/or other materials provided with the distribution.
#     * The names of its contributors may NOT be used to endorse or promote products
#       derived from this software without specific prior written permission.
# 
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL ADRIAN DUSA BE LIABLE FOR ANY
# DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
# ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

#' @name exportDDI
#' @title
#' Export to a DDI metadata file
#'
#' @description
#' Create a DDI Codebook version 2.5, XML file structure.
#'
#' @details
#' #' The information object is essentially a list having two main list components:
#'
#' - \bold{\code{fileDscr}}, if the data is provided in a subcomponent named
#' \bold{\code{datafile}}
#'
#' - \bold{\code{dataDscr}}, having as many components as the number of variables
#' in the (meta)data. For each variable, there should a mandatory subcomponent
#' called \bold{\code{label}} (that contains the variable's label) and, if the
#' variable is of a categorical type, another subcomponent called
#' \bold{\code{values}}.
#'
#' Additional informations about the variables can be specified as further
#' subcomponents, combining DDI specific data but also other information that might
#' not be covered by DDI:
#'
#' - \bold{\code{measurement}} is the equivalent of the specific DDI attribute
#' \bold{\code{nature}} of the \bold{\code{var}} element, and it accepts these
#' values: \code{"nominal"}, \code{"ordinal"}, \code{"interval"}, \code{"ratio"},
#' \code{"percent"}, and \code{"other"}.
#'
#' - \bold{\code{type}} is useful for multiple reasons. A first one, if the
#' variable is numerical, is to differentiate between \code{discrete} and
#' \code{contin} values of the attribute \bold{\code{intrvl}} from the same DDI
#' element \bold{\code{var}}. Another reason is to help identifying pure string
#' variables (containing text), when the subcomponent \bold{\code{type}} is equal
#' to \code{"char"}. It is also used for the subelement \bold{\code{varFormat}} of
#' the element \bold{\code{var}}. Finally, another reason is to differentiate
#' between pure categorical (\code{"cat"}) and pure numerical (\code{"num"})
#' variables, as well as mixed ones, among which \code{"numcat"} referring to a
#' numerical variable with very few values (such as the number of children), for
#' which it is possible to also produce a table of frequencies along the numerical
#' summaries. There are also categorical variables that can be interpreted as
#' numeric (\code{"catnum"}), such as a Likert type response scale with 7 values,
#' where numerical summaries are also routinely performed along with the usual
#' table of frequencies.
#'
#' - \bold{\code{missing}} is an important subcomponent, indicating which of the
#' values in the variable are going to be treated as missing values, and it is
#' going to be exported as the attribute \code{missing} of the DDI subelement
#' \bold{\code{catgry}}.
#'
#' There are many more possible attributes and DDI elements to be added in the
#' information object, future versions of this function will likely expand.
#'
#' For the moment, only DDI codebook version 2.5 is exported, and DDI Lifecycle is
#' planned for future releases.
#'
#' Argument \bold{\code{xmlang}} expects a two letter ISO country coding, for
#' instance \code{"en"} to indicate English, or \code{"ro"} to indicate Romanian etc.
#'
#' If the document is monolang, this argument is placed a single time for the
#' entire document, in the attributes of the \code{codeBook} element. For
#' multilingual documents, it is placed in the attributes of various other
#' (sub)elements, for instance \code{abstract} as an obvious one, or the study
#' title, name of the distributing institution, variable labels etc.
#'
#' The argument \bold{\code{OS}} can be either:\cr
#' \code{"windows"} (default), or \code{"Windows"}, \code{"Win"}, \code{"win"},\cr
#' \code{"MacOS"}, \code{"Darwin"}, \code{"Apple"}, \code{"Mac"}, \code{"mac"},\cr
#' \code{"Linux"}, \code{"linux"}.
#'
#' The end of line separator changes only when the target OS is different from the
#' running OS.
#'
#' The argument \bold{\code{indent}} controls how many spaces will be used in the
#' XML file, to indent the different subelements.
#'
#' A small number of required DDI specific elements and attributes have generic
#' default values but they may be specified using the three dots \bold{\code{...}}
#' argument. For the current version, these are: \code{IDNo}, \code{titl},
#' \code{agency}, \code{URI} (for the \code{holdings} element), \code{distrbtr},
#' \code{abstract} and \code{level} (for the \code{otherMat} element).
#'
#' @return
#' An XML file containing a DDI version 2.5 metadata.
#'
#' @seealso
#' \url{https://ddialliance.org/Specification/DDI-Codebook/2.5/XMLSchema/field_level_documentation.html}
#'
#' @examples
#' codeBook <- list(dataDscr = list(
#' ID = list(
#'     label = "Questionnaire ID",
#'     type = "num",
#'     measurement = "interval"
#' ),
#' V1 = list(
#'     label = "Label for the first variable",
#'     labels = c(
#'         "No"             =  0,
#'         "Yes"            =  1,
#'         "Not applicable" = -97,
#'         "Not answered"   = -99),
#'     na_values = c(-99, -97),
#'     type = "cat",
#'     measurement = "nominal"
#' ),
#' V2 = list(
#'     label = "Label for the second variable",
#'     labels = c(
#'         "Very little"    =  1,
#'         "Little"         =  2,
#'         "So, so"         =  3,
#'         "Much"           =  4,
#'         "Very much"      =  5,
#'         "Don't know"     = -98),
#'     na_values = c(-98),
#'     type = "cat",
#'     measurement = "ordinal"
#' ),
#' V3 = list(
#'     label = "Label for the third variable",
#'     labels = c(
#'         "First answer"   = "A",
#'         "Second answer"  = "B",
#'         "Don't know"     = -98),
#'     na_values = c(-98),
#'     type = "cat",
#'     measurement = "nominal"
#' ),
#' V4 = list(
#'     label = "Number of children",
#'     labels = c(
#'         "Don't know"     = -98,
#'         "Not answered"   = -99),
#'     na_values = c(-99, -98),
#'     type = "numcat",
#'     measurement = "ratio"
#' ),
#' V5 = list(
#'     label = "Political party reference",
#'     type = "char",
#'     txt = "When the respondent indicated his political party reference, his/her
#'         open response was recoded on a scale of 1-99 with parties with a
#'         left-wing orientation coded on the low end of the scale and parties with
#'         a right-wing orientation coded on the high end of the scale. Categories
#'         90-99 were reserved miscellaneous responses."
#' )))
#'
#' \dontrun{
#' exportDDI(codeBook, file = "codebook.xml")
#'
#' # using a namespace
#' exportDDI(codeBook, file = "codebook.xml", xmlns = "ddi")
#' }
#'
#' @author Adrian Dusa
#'
#' @param codebook
#' A list object containing the metadata, or a path to a directory
#' where these objects are located, for batch processing
#'
#' @param file
#' either a character string naming a file or a connection open for
#' writing. "" indicates output to the console
#'
#' @param embed
#' Logical, embed the CSV datafile in the XML file, if present
#'
#' @param OS
#' The target operating system, for the eol - end of line character(s)
#'
#' @param indent Indent width, in number of spaces
#'
#' @param monolang Logical, monolang or multilingual document
#'
#' @param xmlang ISO two letter code for the language used in the DDI elements
#'
#' @param xmlns
#' Character, namespace for the XML file (ignored if already present
#' in the codebook object)
#'
#' @param ... Other arguments, mainly for internal use
#'
#' @export

`exportDDI` <- function(
    codebook, file = "", embed = TRUE, OS = "", indent = 4,
    monolang = FALSE, xmlang = "en", xmlns = "", ...
) {
    # https://ddialliance.org/Specification/DDI-Codebook/2.5/XMLSchema/field_level_documentation.html

    # validation procedure:
    # https://ddialliance.org/Specification/DDI-Codebook/2.5/
    # schema <- read_xml("path/to/ddi_2_5_1/schemas/codebook.xsd")
    # doc <- read_xml("path/to/filetovalidate.xml")
    # xml_validate(doc, schema)

    `check_arg` <- function(x, vx, type = "c") {
        valid <- is.atomic(vx) && length(vx) == 1
        if (valid & type == "c") { # character
            valid <- is.character(vx) & !admisc::possibleNumeric(vx)
        }

        if (!valid) {
            admisc::stopError(
                sprintf(
                    "Argument '%s' should be a%s vector of length 1",
                    x,
                    ifelse(type == "c", " character", "")
                )
            )
        }
    }

    `check_dots` <- function(x, default, type = "c") {
        if (is.element(x, names(dots))) {
            dotsx <- dots[[x]]
            check_arg(x, dotsx, type = type)
            return(dotsx)
        }
        return(default)
    }

    `generateUUID` <- function(x) {
        toreturn <- rep(NA, x)
        first <- sample(c(LETTERS, letters), x, replace = TRUE)
        for (i in seq(x)) {
            toreturn[i] <- paste(
                c(
                    first[i],
                    sample(c(LETTERS, letters, 0:9), 15, replace = TRUE)
                ),
                collapse = ""
            )
        }
        return(toreturn)
        # a safer way is to use unique() but it is highly unlikely this would be needed
        # toreturn <- unique(toreturn)
        # if (length(toreturn) == x) return(toreturn)
    }

    `repeatSpace` <- function(times) {
        paste(rep(" ", times*indent), collapse = "")
    }

    `replaceChars` <- function(x) {
        x <- replaceTicks(x)
        x <- gsub("&", "&amp;", x)
        x <- gsub("<", "&lt;", x)
        x <- gsub(">", "&gt;", x)
        return(x)
    }

    dots <- list(...)

    IDNo <- check_dots("IDNo", "S0000", type = "any")
    titl <- check_dots("titl", "Generic title")
    agency <- check_dots("agency", "default")
    URI <- check_dots("URI", "http://www.default.eu")
    distrbtr <- check_dots("distrbtr", "Name of the distributing institution")
    abstract <- check_dots("abstract", "Study abstract")
    level <- check_dots("level", "0")

    check_arg("xmlang", xmlang)

    xmlang <- paste0(
        ifelse(isTRUE(monolang), "", " "),
        "xml:lang=\"", xmlang, "\""
    )

    ns <- codebook[["xmlns"]]
    if (is.null(ns)) ns <- xmlns
    check_arg("xmlns", ns, "c")

    s0 <- repeatSpace(0)
    s1 <- repeatSpace(1)
    s2 <- repeatSpace(2)
    s3 <- repeatSpace(3)
    s4 <- repeatSpace(4)
    s5 <- repeatSpace(5)

    `catText` <- function(x, ...) {
        cat(paste(
            repeatSpace(x),
            paste(
                unlist(list(...)),
                collapse = ""
            ),
            enter,
            sep = ""
        ))
    }

    if (OS == "") {
        OS <- Sys.info()[["sysname"]]
    }

    enter <- getEnter(OS = OS)

    data <- codebook[["fileDscr"]][["datafile"]]
    stdyDscr <- codebook[["stdyDscr"]]
    dataDscr <- codebook[["dataDscr"]]
    pN <- logical(length(dataDscr))

    # uuid for all variables
    uuid <- generateUUID(length(dataDscr))

    prodate <- as.character(Sys.time())
    version <- as.character(packageVersion("DDIwR"))
    varnames <- names(dataDscr)

    if (!identical(file, "")) {
        sink(file)
        on.exit(sink())
    }

    prodDate <- as.character(Sys.time())
    version <- as.character(packageVersion("DDIwR"))
    varnames <- names(dataDscr)
    cat(paste(
        s0, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>",
        enter,
        sep = ""
    ))

    cat(paste(
        s0, "<", ns, ifelse(identical(ns, ""), "", ":"), "codeBook version=\"2.5\"",
        enter,
        # Apparently, Nesstar interprets this ID as the Study Description ID
        # "ID=\"", generateUUID(1), "\"",
        # enter,
        ifelse(isTRUE(monolang), paste0(xmlang, enter), ""),
        "xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"",
        enter,
        "xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\"",
        enter,
        "xsi:schemaLocation=\"",
        "ddi:codebook:2_5 https://ddialliance.org/Specification/DDI-Codebook/2.5/XMLSchema/codebook.xsd\"",
        enter,
        paste0("xmlns", ifelse(identical(ns, ""), "", ":"), ns, "=\"ddi:codebook:2_5\">"),
        enter,
        sep = ""
    ))

    if (!identical(ns, "")) ns <- paste0(ns, ":")

    if (isTRUE(monolang)) {
        xmlang <- ""
    }

    # Document description
    cat(paste(s1, "<", ns, "docDscr>", enter, sep = ""))
    cat(paste(s2, "<", ns, "citation>", enter, sep = ""))
    cat(paste(s3, "<", ns, "titlStmt>", enter, sep = ""))
    cat(paste(
        s4, "<", ns, "titl", xmlang, ">", titl, "</", ns, "titl>",
        enter,
        sep = ""
    ))
    cat(paste(
        s4, "<", ns, "IDNo agency=\"", agency,"\">", generateUUID(1), "</", ns, "IDNo>",
        enter,
        sep = ""
    ))
    cat(paste(s3, "</", ns, "titlStmt>", enter, sep = ""))
    cat(paste(s3, "<", ns, "prodStmt>", enter, sep = ""))
    cat(paste(
        s4, "<", ns, "prodDate date=\"", prodDate, "\">",
        prodDate, "</", ns, "prodDate>",
        enter,
        sep = ""
    ))
    cat(paste(
        s4, "<", ns, "software version=\"", version,
        "\">R package DDIwR</", ns, "software>",
        enter, sep = ""
    ))
    cat(paste(s3, "</", ns, "prodStmt>", enter, sep = ""))
    cat(paste(s2, "</", ns, "citation>", enter, sep = ""))
    cat(paste(s1, "</", ns, "docDscr>", enter, sep = ""))

    # Study description
    if (is.null(stdyDscr)) {
        cat(paste(s1, "<", ns, "stdyDscr>", enter, sep = ""))
        cat(paste(s2, "<", ns, "citation>", enter, sep = ""))

        cat(paste(s3, "<", ns, "titlStmt>", enter, sep = ""))
        cat(paste(
            s4, "<", ns, "titl", xmlang, ">", titl, "</", ns, "titl>",
            enter,
            sep = ""
        ))
        cat(paste(
            s4, "<", ns, "IDNo agency=\"", agency,"\">", IDNo, "</", ns, "IDNo>",
            enter,
            sep = ""
        ))
        cat(paste(s3, "</", ns, "titlStmt>", enter, sep = ""))

        cat(paste(s3, "<", ns, "distStmt>", enter, sep = ""))
        cat(paste(
            s4, "<", ns, "distrbtr", xmlang, ">", distrbtr,
            "</", ns, "distrbtr>",
            enter, sep = ""
        ))
        cat(paste(s3, "</", ns, "distStmt>", enter, sep = ""))

        cat(paste(
            s3, "<", ns, "holdings URI=\"", URI,
            "\">Description of the study holdings</", ns, "holdings>",
            enter, sep = ""
        ))

        cat(paste(s2, "</", ns, "citation>", enter, sep = ""))
        cat(paste(s2, "<", ns, "stdyInfo>", enter, sep = ""))
        cat(paste(
            s3, "<", ns, "abstract", xmlang, ">", abstract, "</", ns, "abstract>",
            enter, sep = ""
        ))

        cat(paste(s2, "</", ns, "stdyInfo>", enter, sep = ""))

        cat(paste(s1, "</", ns, "stdyDscr>", enter, sep = ""))
    }
    else {

        cat(paste(makeXML(stdyDscr, 1, indent, ns, enter), collapse = ""))

        ## when no xml namespace is needed, this works:
        # stdyDscr <- as_xml_document(list(stdyDscr = stdyDscr))

        # addns <- function(x) {
        #     if (!is.null(names(x))) {
        #         names(x) <- paste0(ns, names(x))
        #         if (is.list(x)) {
        #             x <- lapply(x, function(x) {
        #                 addns(x)
        #             })
        #         }
        #     }
        #     return(x)
        # }

        # stdyDscr <- lapply(list(stdyDscr), addns)
        # names(stdyDscr) <- paste0(ns, "stdyDscr")
        # stdyDscr <- as_xml_document(stdyDscr) # error can not find namespace
        # cat(as.character(stdyDscr))
    }

    fileDscrUUID <- generateUUID(1)
    cat(paste(
        s1, "<", ns, "fileDscr ID=\"", fileDscrUUID, "\">",
        enter,
        sep = ""
    ))

    if (!is.null(data)) {
        if (!is.data.frame(data)) {
            admisc::stopError(
                "The 'datafile' component should be a data frame."
            )
        }
        else if (!identical(toupper(names(data)), toupper(names(dataDscr)))) {
            admisc::stopError(
                "Variables in the data do not match the variables in the data description."
            )
        }

        cat(paste(s2, "<", ns, "fileTxt>", enter, sep = ""))
        if (!is.null(fileName <- codebook[["fileDscr"]][["fileName"]])) {
            cat(paste(s3, "<", ns, "fileName>", fileName, "</", ns, "fileName>", enter, sep = ""))
        }
        cat(paste(s3, "<", ns, "dimensns>", enter, sep = ""))
        cat(paste(s4, "<", ns, "caseQnty>", nrow(data), "</", ns, "caseQnty>", enter, sep = ""))
        cat(paste(s4, "<", ns, "varQnty>", ncol(data), "</", ns, "varQnty>", enter, sep = ""))
        cat(paste(s3, "</", ns, "dimensns>", enter, sep = ""))
        if (!is.null(fileType <- codebook[["fileDscr"]][["fileType"]])) {
            cat(paste(s3, "<", ns, "fileType>", fileType, "</", ns, "fileType>", enter, sep = ""))
        }
        cat(paste(s2, "</", ns, "fileTxt>", enter, sep = ""))

        if (embed) {
            cat(paste(s2, "<", ns, "notes>", enter, sep = ""))
            cat(paste(
                s0, "<![CDATA[# start data #",
                enter,
                sep = ""
            ))

            sink()
            suppressWarnings(
                write.table(
                    undeclare(data, drop = TRUE),
                    file = file,
                    sep = ",",
                    na = "",
                    row.names = FALSE,
                    append = TRUE
                )
            )
            sink(file, append = TRUE)

            cat(paste(
                "# end data #", enter, "]]>",
                enter,
                sep = ""
            ))
            cat(paste(s2, "</", ns, "notes>", enter, sep = ""))
        }

        pN <- sapply(data[names(dataDscr)], function(x) {
            admisc::possibleNumeric(unclass(x))
        })

        aN <- lapply(
            subset(
                data,
                select = is.element(
                    names(data),
                    names(pN)[pN]
                )
            ),
            # data[, names(pN)[pN], drop = FALSE],
            function(x) admisc::asNumeric(unclass(x))
        )

    }

    cat(paste(s1, "</", ns, "fileDscr>", enter, sep = ""))
    cat(paste(s1, "<", ns, "dataDscr>", enter, sep = ""))

    for (i in seq(length(dataDscr))) {
        dcml <- ""
        if (!is.null(data)) {
            dcml <- paste0(
                " dcml=\"",
                ifelse(
                    pN[[names(dataDscr)[i]]],
                    getDecimals(na.omit(aN[[names(dataDscr)[i]]])),
                    0
                ),
                "\""
            )
        }

        nature <- ""
        if(any(grepl("measurement", names(dataDscr[[i]])))) {
            nature <- paste0(" nature=\"", dataDscr[[i]]$measurement, "\"")
        }

        cat(paste0(
            s2, "<", ns, "var ID=\"", uuid[i], "\"",
            " name=\"", varnames[i], "\"",
            " files=\"", fileDscrUUID, "\"",
            dcml, nature, ">",
            enter
        ))

        if (!is.null(dataDscr[[i]][["label"]])) {
            if (!is.na(dataDscr[[i]][["label"]])) {
                cat(paste(
                    s3, "<", ns, "labl", xmlang, ">",
                    replaceChars(
                        dataDscr[[i]][["label"]]
                    ),
                    "</", ns, "labl>",
                    enter,
                    sep = ""
                ))
            }
        }

        na_values <- NULL
        if (is.element("na_values", names(dataDscr[[i]]))) {
            na_values <- dataDscr[[i]]$na_values
        }

        na_range <- NULL
        if (is.element("na_range", names(dataDscr[[i]]))) {
            na_range <- dataDscr[[i]]$na_range
        }

        if (length(na_range) > 0) {
            cat(paste(s3, "<", ns, "invalrng>", enter, sep = ""))

            if (any(is.element(na_range, c(-Inf, Inf)))) {
                if (identical(na_range[1], -Inf)) {
                    cat(paste(
                        s4,
                        sprintf(
                            "<%srange UNITS=\"INT\" max=\"%s\"/>",
                            ns, na_range[2]
                        ),
                        enter,
                        sep = ""
                    ))
                }
                else {
                    cat(paste(
                        s4,
                        sprintf(
                            "<%srange UNITS=\"INT\" min=\"%s\"/>",
                            ns, na_range[1]
                        ),
                        enter,
                        sep = ""
                    ))
                }
            }
            else {
                cat(paste(
                    s4,
                    sprintf(
                        "<%srange UNITS=\"INT\" min=\"%s\" max=\"%s\"/>",
                        ns, na_range[1], na_range[2]
                    ),
                    enter,
                    sep = ""
                ))
            }

            cat(paste(s3, "</", ns, "invalrng>", enter, sep = ""))
        }

        lbls <- dataDscr[[i]][["labels"]]
        if (!is.null(lbls)) {
            nms <- names(lbls)

            # TODO: XML doesn't seem to cope well with multibyte characters?
            # TODO: what about multibyte languages (e.g. Asian)
            multibyte <- grepl("[^!-~ ]", lbls)
            if (any(multibyte)) {
                for (m in which(multibyte)) {
                    strlab <- unlist(strsplit(lbls[m], split = ""))
                    strlab <- strlab[!grepl("[^!-~ ]", strlab)]
                    lbls[m] <- paste(strlab, collapse = "")
                }
            }

            lbls <- setNames(admisc::trimstr(lbls), nms)
            # names(lbls) <- nms
        }

        type <- dataDscr[[i]]$type

        # even if the data is not present, pN is FALSE for all variables
        if (pN[i]) {
            vals <- aN[[names(dataDscr)[i]]]

            if (!is.null(lbls)) {
                ismiss <- is.element(lbls, na_values)
                if (length(na_range) > 0) {
                    ismiss <- ismiss | (lbls >= na_range[1] & lbls <= na_range[2])
                }
                vals[is.element(vals, lbls[ismiss])] <- NA
            }

            vals <- na.omit(vals)

            # this is a test if a variable truly is numeric
            # (not just categorical using numbers)
            # if it has at least four(?) values different from the labels
            printnum <- length(setdiff(vals, lbls)) > 4

            if (!is.null(type)) {
                # at least two non missing values are needed to calculate sd()
                printnum <- printnum | (length(vals) > 2 & grepl("num", type))
            }

            if (printnum) { # numeric variable
                cat(paste(
                    s3,
                    "<", ns, "sumStat type=\"min\">",
                    format(
                        min(vals, na.rm = TRUE),
                        scientific = FALSE
                    ),
                    "</", ns, "sumStat>",
                    enter,
                    sep = ""
                ))

                cat(paste(
                    s3,
                    "<", ns, "sumStat type=\"max\">",
                    format(
                        max(vals, na.rm = TRUE),
                        scientific = FALSE
                    ),
                    "</", ns, "sumStat>",
                    enter,
                    sep = ""
                ))

                cat(paste(
                    s3,
                    "<", ns, "sumStat type=\"mean\">",
                    format(
                        mean(vals, na.rm = TRUE),
                        scientific = FALSE
                    ),
                    "</", ns, "sumStat>",
                    enter,
                    sep = ""
                ))

                cat(paste(
                    s3,
                    "<", ns, "sumStat type=\"medn\">",
                    format(
                        median(vals, na.rm = TRUE),
                        scientific = FALSE
                    ),
                    "</", ns, "sumStat>",
                    enter,
                    sep = ""
                ))

                cat(paste(
                    s3,
                    "<", ns, "sumStat type=\"stdev\">",
                    format(
                        sd(vals, na.rm = TRUE),
                        scientific = FALSE
                    ),
                    "</", ns, "sumStat>",
                    enter,
                    sep = ""
                ))

            }
        }

        if (!is.null(lbls)) {

            tbl <- table(data[[names(dataDscr)[i]]]) # what is the difference from data[[i]] ?

            for (v in seq(length(lbls))) {

                ismiss <- is.element(lbls[v], na_values)
                if (length(na_range) > 0) {
                    ismiss <- ismiss | (lbls[v] >= na_range[1] & lbls[v] <= na_range[2])
                }

                cat(paste(
                    s3,
                    "<", ns, "catgry", ifelse(ismiss, " missing=\"Y\"", ""), ">",
                    enter,
                    sep = ""
                ))

                cat(paste(
                    s4,
                    "<", ns, "catValu>",
                    lbls[v],
                    "</", ns, "catValu>",
                    enter,
                    sep = ""
                ))

                cat(paste(
                    s4,
                    "<", ns, "labl", xmlang, ">",
                    replaceChars(names(lbls)[v]),
                    "</", ns, "labl>",
                    enter,
                    sep = ""
                ))

                if (!is.null(data)) {
                    freq <- tbl[match(lbls[v], names(tbl))]
                    cat(paste(
                        s4,
                        "<", ns, "catStat type=\"freq\">",
                        ifelse(
                            is.na(freq),
                            0,
                            format(freq, scientific = FALSE)
                        ),
                        "</", ns, "catStat>",
                        enter,
                        sep = ""
                    ))
                }

                cat(paste(s3, "</", ns, "catgry>", enter, sep = ""))
            }
        }

        if (any(grepl("type", names(dataDscr[[i]])))) {
            varFormat <- dataDscr[[i]]$varFormat[1] # SPSS
            cat(paste(
                s3,
                "<", ns, "varFormat type=\"",
                ifelse(
                    grepl("char", dataDscr[[i]]$type),
                    "character",
                    "numeric"
                ),
                "\" schema=\"other\" formatname=\"",
                substr(varFormat, 1, 1),
                "\">",
                varFormat,
                "</", ns, "varFormat>",
                enter,
                sep = ""
            ))
        }

        if (any(grepl("txt", names(dataDscr[[i]])))) {
            cat(paste(s3, "<", ns, "txt>", enter, sep = ""))
            cat(paste(
                s0,
                "<![CDATA[", dataDscr[[i]]$txt, "]]>",
                enter,
                sep = ""
            ))
            cat(paste(s3, "</", ns, "txt>", enter, sep = ""))
        }

        cat(paste(s2, "</", ns, "var>", enter, sep = ""))
    }

    cat(paste(s1, "</", ns, "dataDscr>", enter, sep = ""))
    cat(paste(s1, "<", ns, "otherMat level=\"", level, "\"></", ns, "otherMat>", enter, sep = ""))
    cat(paste(s0, "</", ns, "codeBook>", enter, sep = ""))

}
