\name{ccd.augment}
\alias{ccd.augment}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{ Functions for accessing central composite designs from package rsm }
\description{
  Functions for accessing central composite designs from package rsm, 
  and for augmenting existing fractional factorials in case of a late decision 
  for a frequential procedure. 
}
\usage{
ccd.augment(cube, ncenter = 4, columns="all", block.name="Block.ccd",
        alpha = "orthogonal", randomize=TRUE, seed=NULL, ...)
}
\arguments{
  \item{cube}{design generated by function \code{\link[FrF2]{FrF2}}. The design 
        must not be a split-plot design, nor a parameter design in long version. }
  \item{ncenter}{ integer number of center points for each cube or star point block, 
        or vector with two numbers, the first for the cube and the second for 
        the star portion of the design }
  \item{block.name}{ name of block factor that distinguishes (at least) between blocks; 
        even for unblocked cubes, the ccd design has a cube and a star point block }
  \item{alpha}{ \dQuote{orthogonal}, \dQuote{rotatable}, 
        or a number that indicates the position of the star points;
        the number 1 would create a face-centered design. }
  \item{randomize}{ logical that indicates whether or not randomization should occur }
  \item{seed}{ NULL or a vector of two integer seeds for random number generation in randomization }
  \item{\dots}{ reserved for future usage }
  \item{columns}{not yet implemented; 
        it is intended to later allow to add star points for only some factors 
        of a design (after eliminating the others as unimportant in a sequential process), 
        and columns will be used to indicate those}
}
\details{
    The statistical background of central composite designs is briefly described 
    under \code{\link{CentralCompositeDesigns}}.
    
   Function \code{ccd.augment} augments an existing 2-level fractional factorial 
   that should already have been run with center points and should have resolution V. 
  
   In exceptional situations, it may be useful to base a ccd on a resolution IV design 
   that allows estimation of all 2-factor interactions of interest. Thus, it can be 
   interesting to apply function \code{ccd.augment} to a cube 
   based on the \code{estimable} functionality of function \code{\link[FrF2]{FrF2}} 
   in cases where a resolution V cube is not feasible.
   Of course, this does not allow to estimate the aliased 2-factor interactions 
   and therefore generates a warning.
   }
\value{
    The function returns a data frame of S3 class \code{\link[DoE.base]{design}} 
    with attributes attached. The data frame itself is in the original data scale. 
    The data frame \code{desnum} attached as attribute \code{desnum} is the original data frame 
    returned by package \code{rsm}. The attribute \code{design.info} is a list of various design properties. 
    The element \code{type} of that list is the character string \code{ccd}.
    Besides the elements present in all class \code{\link[DoE.base]{design}} objects, 
    there are the elements quantitative (vector with \code{nfactor} TRUE entries),
    and a \code{codings} element usable in the coding functions available in the \pkg{rsm} 
    package, e.g. \code{\link[rsm]{coded.data}}.
    
    Note that the row names and the standard order column in the \code{run.order} 
    attribute of ccd designs based on 
    estimability requirements (cf. also the details section) are not in conventional order 
    and should not be used as the basis for any calculations. The same is true for 
    blocked designs, if the blocking routine \code{\link[FrF2]{blockpick.big}} was used.
    }
\references{ 
  Box, G.E.P., Hunter, J.S. and Hunter, W.G. (2005, 2nd ed.). \emph{Statistics for Experimenters}. 
    Wiley, New York.
     
  Box, G.E.P. and Wilson, K.B. (1951). On the Experimental Attainment of Optimum Conditions. 
    \emph{J. Royal Statistical Society}, \bold{B13}, 1-45.
    
  NIST/SEMATECH e-Handbook of Statistical Methods, 
    \url{http://www.itl.nist.gov/div898/handbook/pri/section3/pri3361.htm}, 
    accessed August 20th, 2009.
    
  Myers, R.H., Montgomery, D.C. and Anderson-Cook, C.M. (2009). \emph{Response Surface Methodology. 
      Process and Product Optimization Using Designed Experiments}. Wiley, New York.
}
\author{ Ulrike Groemping }
\note{ This package is currently under intensive development. Substantial changes are to be expected in the near future.}
\examples{
  ## purely technical examples for the sequential design creation process
    ## start with a fractional factorial with center points
    plan <- FrF2(16,5,default.levels=c(10,30),ncenter=6)
    ## collect data and add them to the design
    y <- rexp(22)
    plan <- add.response(plan,y)
    ## assuming that an analysis has created the suspicion that a second order 
    ## model should be fitted (not to be expected for the above random numbers):
    plan.augmented <- ccd.augment(plan, ncenter=4)
    ## add new responses to the design
    y <- c(y, rexp(14))  ## append responses for the 14=5*2 + 4 star points
    r.plan.augmented <- add.response(plan.augmented, y, replace=TRUE)

  ## for info: how to analyse results from such a desgin
    lm.result <- lm(y~Block.ccd+(.-Block.ccd)^2+I(A^2)+I(B^2)+I(C^2)+I(D^2)+I(E^2), r.plan.augmented)
    summary(lm.result)
    ## analysis with function rsm
    rsm.result <- rsm(y~Block.ccd+SO(A,B,C,D,E), r.plan.augmented)
    summary(rsm.result)  ## provides more information than lm.result
    loftest(rsm.result)  ## separate lack of fit test
    ## graphical analysis 
    ## (NOTE: purely for demo purposes, the model is meaningless here)
    ## individual contour plot
    contour(rsm.result,B~A)
    ## several contour plots
    par(mfrow=c(1,2))
    contour(rsm.result,list(B~A, C~E))
    ## many contourplots, all pairs of some factors
    par(mfrow=c(2,3))
    contour(rsm.result,~A+B+C+D)
}
\seealso{ ~~See also , \code{\link{ccd.design}}, \code{\link[FrF2]{FrF2}}, \code{\link[lhs]{lhs}}, \code{\link[rsm]{rsm}} }
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{ array }
\keyword{ design }% __ONLY ONE__ keyword per line
