% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Q_rwl.R
\name{rwl}
\alias{rwl}
\title{Residual Weighted Learning}
\usage{
rwl(
  ...,
  moPropen,
  moMain,
  data,
  reward,
  txName,
  regime,
  response,
  fSet = NULL,
  lambdas = 2,
  cvFolds = 0L,
  kernel = "linear",
  kparam = NULL,
  responseType = "continuous",
  verbose = 2L
)
}
\arguments{
\item{...}{Used primarily to require named input. However, inputs for
the optimization methods can be sent through the ellipsis. The
optimization method is stats::optim().}

\item{moPropen}{An object of class modelObj or modelObjSubset, which 
defines the model and
R methods to be used to obtain parameter estimates and
predictions for the propensity for treatment.
See ?moPropen for details.}

\item{moMain}{An object of class modelObj or modelObjSubset, which 
defines the model and
R methods to be used to obtain parameter estimates and
predictions for the main effects of the outcome.
See ?modelObj for details.}

\item{data}{A data frame of the covariates and tx histories}

\item{reward}{The response vector}

\item{txName}{A character object.
The column header of \emph{data} that corresponds to the tx covariate}

\item{regime}{A formula object or a list of formula objects.
The covariates to be included in classification. If a list is provided,
this specifies that there is an underlying subset structure -- fSet must
then be defined.}

\item{response}{A numeric vector.
The reward. Allows for naming convention followed in most
DynTxRegime methods.}

\item{fSet}{A function or NULL defining subset structure}

\item{lambdas}{A numeric object or a numeric vector object giving the
penalty tuning parameter. If more than 1 is provided,
the finite set of values to be considered in the
cross-validation algorithm}

\item{cvFolds}{If cross-validation is to be used to select the tuning
parameters, the number of folds.}

\item{kernel}{A character object.
must be one of \{"linear", "poly", "radial"\}}

\item{kparam}{A numeric object of NULL.
If kernel = linear, kparam is ignored.
If kernel = poly, kparam is the degree of the polynomial
If kernel = radial, kparam is the inverse bandwidth of the
kernel. If a vector of bandwidth parameters is given,
cross-validation will be used to select the parameter}

\item{responseType}{A character indicating if response is continuous, binary
or count data.}

\item{verbose}{An integer or logical. If 0, no screen prints are generated. If 1,
screen prints are generated with the exception of optimization results
obtained in iterative algorithm. If 2, all screen prints are generated.}
}
\value{
an RWL object
}
\description{
Residual Weighted Learning
}
\examples{
\dontrun{ 
# Load and process data set
data(bmiData)

# define the negative 12 month change in BMI from baseline
y12 <- -100*(bmiData[,6L] - bmiData[,4L])/bmiData[,4L]

# propensity model
moPropen <- buildModelObj(model = ~parentBMI+month4BMI,
                          solver.method = 'glm',
                          solver.args = list('family'='binomial'),
                          predict.method = 'predict.glm',
                          predict.args = list(type='response'))

# outcome model
moMain <- buildModelObj(model = ~parentBMI+month4BMI,
                        solver.method = 'lm')

fitRWL <- rwl(moPropen = moPropen, moMain = moMain,
              data = bmiData, reward = y12,  txName = 'A2', 
              regime = ~ parentBMI + month4BMI,
              kernel = 'radial', kparam = 1.5)

##Available methods

  # Coefficients of the regression objects
  coef(fitRWL)

  # Description of method used to obtain object
  DTRstep(fitRWL)

  # Estimated value of the optimal treatment regime for training set
  estimator(fitRWL)

  # Value object returned by regression methods
  fitObject(fitRWL)

  # Summary of optimization routine
  optimObj(fitRWL)

  # Estimated optimal treatment for training data
  optTx(fitRWL)

  # Estimated optimal treatment for new data
  optTx(fitRWL, bmiData)

  # Value object returned by outcome regression method
  outcome(fitRWL)

  # Plots if defined by regression methods
  dev.new()
  par(mfrow = c(2,4))

  plot(fitRWL)
  plot(fitRWL, suppress = TRUE)

  # Value object returned by propensity score regression method
  propen(fitRWL)

  # Parameter estimates for decision function
  regimeCoef(fitRWL)

  # Show main results of method
  show(fitRWL)

  # Show summary results of method
  summary(fitRWL)
 }
}
\references{
Xin Zhou, Nicole Mayer-Hamblett, Umer Khan, and Michael R Kosorok
   (2017)
   Residual weighted learning for estimating individualized
   treatment rules. Journal of the American Statistical Association,
   112, 169--187.
}
\seealso{
Other statistical methods: 
\code{\link{bowl}()},
\code{\link{earl}()},
\code{\link{iqLearn}},
\code{\link{optimalClass}()},
\code{\link{optimalSeq}()},
\code{\link{owl}()},
\code{\link{qLearn}()}

Other weighted learning methods: 
\code{\link{bowl}()},
\code{\link{earl}()},
\code{\link{owl}()}

Other single decision point methods: 
\code{\link{earl}()},
\code{\link{optimalClass}()},
\code{\link{optimalSeq}()},
\code{\link{owl}()},
\code{\link{qLearn}()}
}
\concept{single decision point methods}
\concept{statistical methods}
\concept{weighted learning methods}
